/*
 * Copyright 2020-2023 Siphalor
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.siphalor.amecs.impl;

//- import com.mojang.blaze3d.platform.Window;
import de.siphalor.amecs.api.KeyBindingUtils;
import de.siphalor.amecs.api.KeyModifier;
import de.siphalor.amecs.api.KeyModifiers;
import de.siphalor.amecs.api.PriorityKeyBinding;
import de.siphalor.amecs.impl.duck.IKeyBinding;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_304;
import net.minecraft.class_310;
import net.minecraft.class_3675;
import org.jetbrains.annotations.ApiStatus;

import java.util.*;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Environment(EnvType.CLIENT)
@ApiStatus.Internal
public class KeyBindingManager {
	// split it in two maps because it is ways faster to only stream the map with the objects we need
	// rather than streaming all and throwing out a bunch every time
	public static final Map<class_3675.class_306, List<class_304>> keysById = new HashMap<>();
	public static final Map<class_3675.class_306, List<class_304>> priorityKeysById = new HashMap<>();private static final List<class_304> pressedKeyBindings = new ArrayList<>(10);

	private KeyBindingManager() {}
	/**
	 * Removes a key binding from one of the internal maps
	 * @param targetMap the key binding map to remove from
	 * @param keyBinding the key binding to remove
	 * @return whether the keyBinding was removed. It is not removed if it was not contained
	 */
	private static boolean removeKeyBindingFromMap(Map<class_3675.class_306, List<class_304>> targetMap, class_304 keyBinding) {
		// we need to get the backing list to remove elements thus we can not use any of the other methods that return streams
		class_3675.class_306 keyCode = ((IKeyBinding) keyBinding).amecs$getBoundKey();
		List<class_304> keyBindings = targetMap.get(keyCode);
		if (keyBindings == null) {
			return false;
		}
		boolean removed = false;
		// while loop to ensure that we remove all equal KeyBindings if for some reason there should be duplicates
		while (keyBindings.remove(keyBinding)) {
			removed = true;
		}
		return removed;
	}

	/**
	 * Adds a key binding to one of the internal maps
	 * @param targetMap the key binding map to add to
	 * @param keyBinding the key binding to add
	 * @return whether the keyBinding was added. It is not added if it is already contained
	 */
	private static boolean addKeyBindingToListFromMap(Map<class_3675.class_306, List<class_304>> targetMap, class_304 keyBinding) {
		class_3675.class_306 keyCode = ((IKeyBinding) keyBinding).amecs$getBoundKey();
		List<class_304> keyBindings = targetMap.computeIfAbsent(keyCode, k -> new ArrayList<>());
		if (keyBindings.contains(keyBinding)) {
			return false;
		}
		keyBindings.add(keyBinding);
		return true;
	}

	/**
	 * Registers a key binding to Amecs API
	 * @param keyBinding the key binding to register
	 * @return whether the keyBinding was added. It is not added if it is already contained
	 */
	public static boolean register(class_304 keyBinding) {
		if (keyBinding instanceof PriorityKeyBinding) {
			return addKeyBindingToListFromMap(priorityKeysById, keyBinding);
		} else {
			return addKeyBindingToListFromMap(keysById, keyBinding);
		}
	}

	public static Stream<class_304> getMatchingKeyBindings(class_3675.class_306 keyCode, boolean priority) {
		List<class_304> keyBindingList = (priority ? priorityKeysById : keysById).get(keyCode);
		if (keyBindingList == null)
			return Stream.empty();
		// If there are two key bindings, alt + y and shift + alt + y, and you press shift + alt + y, both will be triggered.
		// This is intentional.
		Stream<class_304> result = keyBindingList.stream().filter(KeyBindingManager::areExactModifiersPressed);
		List<class_304> keyBindings = result.collect(Collectors.toList());
		if (keyBindings.isEmpty())
			return keyBindingList.stream().filter(keyBinding -> ((IKeyBinding) keyBinding).amecs$getKeyModifiers().isUnset());
		return keyBindings.stream();
	}

	private static boolean areExactModifiersPressed(class_304 keyBinding) {
		return KeyBindingUtils.getBoundModifiers(keyBinding).equals(AmecsAPI.CURRENT_MODIFIERS);
	}

	public static void onKeyPressed(class_3675.class_306 keyCode) {
		getMatchingKeyBindings(keyCode, false).forEach(keyBinding ->
			((IKeyBinding) keyBinding).amecs$incrementTimesPressed()
		);
	}

	private static Stream<class_304> getKeyBindingsFromMap(Map<class_3675.class_306, List<class_304>> keysById_map) {
		return keysById_map.values().stream().flatMap(Collection::stream);
	}

	private static void forEachKeyBinding(Consumer<class_304> consumer) {
		getKeyBindingsFromMap(priorityKeysById).forEach(consumer);
		getKeyBindingsFromMap(keysById).forEach(consumer);
	}

	private static void forEachKeyBindingWithKey(class_3675.class_306 key, Consumer<class_304> consumer) {
		getMatchingKeyBindings(key, true).forEach(consumer);
		getMatchingKeyBindings(key, false).forEach(consumer);
	}

	public static void updatePressedStates() {
		//# if MC_VERSION_NUMBER >= 12109
		//- Window windowHandle = Minecraft.getInstance().getWindow();
		//# elif MC_VERSION_NUMBER >= 11500
		long windowHandle = class_310.method_1551().method_22683().method_4490();
		//# else
		//- long windowHandle = Minecraft.getInstance().window.getWindow();
		//# end
		forEachKeyBinding(keyBinding -> {
			class_3675.class_306 key = ((IKeyBinding) keyBinding).amecs$getBoundKey();
			boolean pressed = !keyBinding.method_1415() && key.method_1442() == class_3675.class_307.field_1668 && class_3675.method_15987(windowHandle, key.method_1444());
			setKeyBindingPressed(keyBinding, pressed);
		});
	}

	/**
	 * Unregisters a key binding from Amecs API
	 * @param keyBinding the key binding to unregister
	 * @return whether the keyBinding was removed. It is not removed if it was not contained
	 */
	public static boolean unregister(class_304 keyBinding) {
		if (keyBinding == null) {
			return false;
		}
		// avoid having to rebuild the whole entry map with KeyMapping.updateKeysByCode()
		boolean removed = false;
		removed |= removeKeyBindingFromMap(keysById, keyBinding);
		removed |= removeKeyBindingFromMap(priorityKeysById, keyBinding);
		return removed;
	}

	public static void updateKeysByCode() {
		keysById.clear();
		priorityKeysById.clear();
		KeyBindingUtils.getIdToKeyBindingMap().values().forEach(KeyBindingManager::register);
	}

	public static void setKeyBindingPressed(class_304 keyBinding, boolean pressed) {
		if (pressed != keyBinding.method_1434()) {
			if (pressed) {
				pressedKeyBindings.add(keyBinding);
			} else {
				pressedKeyBindings.remove(keyBinding);
			}
		}
		//# if MC_VERSION_NUMBER >= 11500
		keyBinding.method_23481(pressed);
		//# else
		//- ((IKeyBinding) keyBinding).amecs$setDown(pressed);
		//# end
	}

	public static void unpressAll() {
		KeyBindingUtils.getIdToKeyBindingMap().values().forEach(keyBinding -> ((IKeyBinding) keyBinding).amecs$reset());
	}

	public static boolean onKeyPressedPriority(class_3675.class_306 keyCode) {
		// because streams are lazily evaluated, this code only calls onPressedPriority so often until one returns true
		Optional<class_304> keyBindings = getMatchingKeyBindings(keyCode, true).filter(keyBinding -> ((PriorityKeyBinding) keyBinding).onPressedPriority()).findFirst();
		return keyBindings.isPresent();
	}

	public static boolean onKeyReleasedPriority(class_3675.class_306 keyCode) {
		// because streams are lazily evaluated, this code only calls onPressedPriority so often until one returns true
		Optional<class_304> keyBindings = getMatchingKeyBindings(keyCode, true).filter(keyBinding -> ((PriorityKeyBinding) keyBinding).onReleasedPriority()).findFirst();
		return keyBindings.isPresent();
	}

	public static void setKeyPressed(class_3675.class_306 keyCode, boolean pressed) {
		KeyModifier modifier = KeyModifier.fromKeyCode(keyCode.method_1444());
		AmecsAPI.CURRENT_MODIFIERS.set(modifier, pressed);

		// Update keybindings with matching modifiers and the same keycode
		forEachKeyBindingWithKey(keyCode, keyBinding -> setKeyBindingPressed(keyBinding, pressed));

		if (modifier != null && !pressed) {
			handleReleasedModifier();
		}
	}

	private static void handleReleasedModifier() {
		// Handle the case that a modifier has been released
		pressedKeyBindings.removeIf(pressedKeyBinding -> {
			KeyModifiers boundModifiers = KeyBindingUtils.getBoundModifiers(pressedKeyBinding);
			if (!AmecsAPI.CURRENT_MODIFIERS.contains(boundModifiers)) {
				//# if MC_VERSION_NUMBER >= 11500
				pressedKeyBinding.method_23481(false);
				//# else
				//- ((IKeyBinding) pressedKeyBinding).amecs$setDown(false);
				//# end
				return true;
			}
			return false;
		});
	}
}
