/*
 * Copyright 2020 Siphalor
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.siphalor.amecs.key_modifiers.impl;

import de.siphalor.amecs.key_modifiers.api.AmecsKeyModifier;
import de.siphalor.amecs.key_modifiers.api.AmecsKeyModifierCombination;
import de.siphalor.amecs.key_modifiers.api.AmecsKeyModifiers;
import de.siphalor.amecs.key_modifiers.api.AmecsKeyModifiersApi;
import de.siphalor.amecs.key_modifiers.impl.duck.IKeyMapping;
import java.util.*;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_1041;
import net.minecraft.class_304;
import net.minecraft.class_310;
import net.minecraft.class_3675;
import org.jetbrains.annotations.ApiStatus;

@Environment(EnvType.CLIENT)
@ApiStatus.Internal
public class AmecsKeyMappingManager {
	private static final List<AmecsKeyMappingManagerLayer> LAYERS = new ArrayList<>();
	private static final List<class_304> PRESSED_MAPPINGS = new ArrayList<>(10);

	static {
		LAYERS.add(new AmecsKeyMappingManagerLayer());
	}

	private AmecsKeyMappingManager() {}

	public static void prependLayer(AmecsKeyMappingManagerLayer layer) {
		//noinspection SequencedCollectionMethodCanBeUsed
		LAYERS.add(0, layer);
	}

	/**
	 * Registers a key binding to Amecs API
	 * @param mapping the key binding to register
	 * @return whether the keyBinding was added. It is not added if it is already contained
	 */
	public static boolean register(class_304 mapping) {
		for (AmecsKeyMappingManagerLayer layer : LAYERS) {
			if (layer.accepts(mapping)) {
				return layer.register(mapping);
			}
		}
		return false;
	}

	public static void onKeyPressed(class_3675.class_306 input) {
		for (AmecsKeyMappingManagerLayer layer : LAYERS) {
			layer.getMappingsForInput(input).forEach(keyBinding ->
					((IKeyMapping) keyBinding).amecs$incrementTimesPressed()
			);
		}
	}

	public static void updatePressedStates() {
		//# if MC_VERSION_NUMBER >= 12109
		class_1041 windowHandle = class_310.method_1551().method_22683();
		//# elif MC_VERSION_NUMBER >= 11500
		//- long windowHandle = Minecraft.getInstance().getWindow().getWindow();
		//# else
		//- long windowHandle = Minecraft.getInstance().window.getWindow();
		//# end
		for (AmecsKeyMappingManagerLayer layer : LAYERS) {
			layer.getAllMappings().forEach(keyBinding -> {
				class_3675.class_306 key = ((IKeyMapping) keyBinding).amecs$getBoundKey();
				boolean pressed = !keyBinding.method_1415() && key.method_1442() == class_3675.class_307.field_1668 && class_3675.method_15987(windowHandle, key.method_1444());
				setKeyBindingPressed(keyBinding, pressed);
			});
		}
	}

	/**
	 * Unregisters a key binding from Amecs API
	 * @param keyBinding the key binding to unregister
	 * @return whether the keyBinding was removed. It is not removed if it was not contained
	 */
	public static boolean unregister(class_304 keyBinding) {
		if (keyBinding == null) {
			return false;
		}
		// avoid having to rebuild the whole entry map with KeyMapping.updateKeysByCode()
		boolean removed = false;
		for (AmecsKeyMappingManagerLayer layer : LAYERS) {
			removed |= layer.unregister(keyBinding);
		}
		return removed;
	}

	public static void updateKeysByCode() {
		LAYERS.forEach(AmecsKeyMappingManagerLayer::clear);
		AmecsKeyModifiersModule.getIdToKeyBindingMap().values().forEach(AmecsKeyMappingManager::register);
	}

	public static void setKeyBindingPressed(class_304 keyBinding, boolean pressed) {
		if (pressed != keyBinding.method_1434()) {
			if (pressed) {
				PRESSED_MAPPINGS.add(keyBinding);
			} else {
				PRESSED_MAPPINGS.remove(keyBinding);
			}
		}
		//# if MC_VERSION_NUMBER >= 11500
		keyBinding.method_23481(pressed);
		//# else
		//- ((IKeyMapping) keyBinding).amecs$setDown(pressed);
		//# end
	}

	public static void unpressAll() {
		AmecsKeyModifiersModule.getIdToKeyBindingMap().values().forEach(keyBinding -> ((IKeyMapping) keyBinding).amecs$reset());
	}

	public static void setKeyPressed(class_3675.class_306 keyCode, boolean pressed) {
		AmecsKeyModifier modifier = AmecsKeyModifiers.fromKeyCode(keyCode.method_1444());
		if (modifier != null) {
			AmecsKeyModifiersModule.CURRENT_MODIFIERS.set(modifier, pressed);
		}

		// Update keybindings with matching modifiers and the same keycode
		for (AmecsKeyMappingManagerLayer layer : LAYERS) {
			layer.getMappingsForInput(keyCode).forEach(keyBinding -> setKeyBindingPressed(keyBinding, pressed));
		}

		if (modifier != null && !pressed) {
			handleReleasedModifier();
		}
	}

	private static void handleReleasedModifier() {
		// Handle the case that a modifier has been released
		PRESSED_MAPPINGS.removeIf(pressedKeyBinding -> {
			AmecsKeyModifierCombination boundModifiers = AmecsKeyModifiersApi.getBoundModifiers(pressedKeyBinding);
			if (!AmecsKeyModifiersModule.CURRENT_MODIFIERS.contains(boundModifiers)) {
				//# if MC_VERSION_NUMBER >= 11500
				pressedKeyBinding.method_23481(false);
				//# else
				//- ((IKeyMapping) pressedKeyBinding).amecs$setDown(false);
				//# end
				return true;
			}
			return false;
		});
	}
}
