/*
 * Copyright 2020-2022 Siphalor
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 */

package de.siphalor.nbtcrafting;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.mojang.datafixers.util.Pair;
import io.netty.buffer.Unpooled;
import it.unimi.dsi.fastutil.ints.*;
import net.fabricmc.api.ModInitializer;
import net.fabricmc.fabric.api.networking.v1.ServerLoginConnectionEvents;
import net.fabricmc.fabric.api.networking.v1.ServerLoginNetworking;
import net.fabricmc.fabric.api.networking.v1.ServerPlayConnectionEvents;
import net.minecraft.class_1263;
import net.minecraft.class_1662;
import net.minecraft.class_1856;
import net.minecraft.class_1860;
import net.minecraft.class_1863;
import net.minecraft.class_1865;
import net.minecraft.class_2378;
import net.minecraft.class_2487;
import net.minecraft.class_2540;
import net.minecraft.class_2960;
import net.minecraft.class_3222;
import net.minecraft.class_3956;
import net.minecraft.recipe.*;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.siphalor.nbtcrafting.advancement.StatChangedCriterion;
import de.siphalor.nbtcrafting.api.RecipeTypeHelper;
import de.siphalor.nbtcrafting.ingredient.IIngredient;
import de.siphalor.nbtcrafting.mixin.advancement.MixinCriterions;
import de.siphalor.nbtcrafting.recipe.AnvilRecipe;
import de.siphalor.nbtcrafting.recipe.BrewingRecipe;
import de.siphalor.nbtcrafting.recipe.IngredientRecipe;
import de.siphalor.nbtcrafting.recipe.WrappedRecipeSerializer;
import de.siphalor.nbtcrafting.recipe.cauldron.CauldronRecipe;
import de.siphalor.nbtcrafting.recipe.cauldron.CauldronRecipeSerializer;
import de.siphalor.nbtcrafting.util.duck.IServerPlayerEntity;

public class NbtCrafting implements ModInitializer {
	public static final String MOD_ID = "nbtcrafting";
	public static final String MOD_NAME = "Nbt Crafting";

	private static final String LOG_PREFIX = "[" + MOD_NAME + "] ";
	private static final Logger LOGGER = LogManager.getLogger();

	public static final class_2960 PRESENCE_CHANNEL = new class_2960(MOD_ID, "present");
	public static final class_2960 UPDATE_ANVIL_TEXT_S2C_PACKET_ID = new class_2960(MOD_ID, "update_anvil_text");
	public static final class_2960 UPDATE_ADVANCED_RECIPES_PACKET_ID = new class_2960(MOD_ID, "update_advanced_recipes");

	public static final class_3956<AnvilRecipe> ANVIL_RECIPE_TYPE = registerRecipeType("anvil");
	@SuppressWarnings("unused")
	public static final class_1865<AnvilRecipe> ANVIL_RECIPE_SERIALIZER = registerRecipeSerializer("anvil", AnvilRecipe.SERIALIZER);

	public static final class_3956<BrewingRecipe> BREWING_RECIPE_TYPE = registerRecipeType("brewing");
	@SuppressWarnings("unused")
	public static final class_1865<BrewingRecipe> BREWING_RECIPE_SERIALIZER = registerRecipeSerializer("brewing", BrewingRecipe.SERIALIZER);

	public static final class_3956<CauldronRecipe> CAULDRON_RECIPE_TYPE = registerRecipeType("cauldron");
	public static final CauldronRecipeSerializer CAULDRON_RECIPE_SERIALIZER = registerRecipeSerializer("cauldron", new CauldronRecipeSerializer());

	public static final class_3956<IngredientRecipe<class_1263>> SMITHING_RECIPE_TYPE = registerRecipeType("smithing");
	@SuppressWarnings("unused")
	public static final class_1865<IngredientRecipe<class_1263>> SMITHING_RECIPE_SERIALIZER = registerRecipeSerializer("smithing", new IngredientRecipe.Serializer<>((id, base, ingredient, result, serializer) -> new IngredientRecipe<>(id, base, ingredient, result, SMITHING_RECIPE_TYPE, serializer)));

	public static final class_1865<class_1860<?>> WRAPPED_RECIPE_SERIALIZER = registerRecipeSerializer("wrapped", new WrappedRecipeSerializer());

	public static final StatChangedCriterion STAT_CHANGED_CRITERION = MixinCriterions.registerCriterion(new StatChangedCriterion());

	private static boolean lastReadNbtPresent = false;
	private static class_2487 lastReadNbt;

	public static class_1662 lastRecipeFinder;
	public static ThreadLocal<class_3222> lastServerPlayerEntity = new ThreadLocal<>();
	public static ThreadLocal<Boolean> advancedIngredientSerializationEnabled = new ThreadLocal<>();
	private static final IntSet hasModClientConnectionHashes = IntSets.synchronize(new IntAVLTreeSet());

	private static int currentStackId = 1;
	public static final Int2ObjectMap<Pair<Integer, class_2487>> id2StackMap = new Int2ObjectAVLTreeMap<>();
	public static final LoadingCache<Pair<Integer, class_2487>, Integer> stack2IdMap = CacheBuilder.newBuilder().expireAfterAccess(5, TimeUnit.MINUTES).removalListener(notification -> {
				synchronized (id2StackMap) {
					id2StackMap.remove((int) notification.getValue());
				}
			}
	).build(new CacheLoader<>() {
		@Override
		public Integer load(Pair<Integer, NbtCompound> key) {
			synchronized (id2StackMap) {
				id2StackMap.put(currentStackId, key);
			}
			return currentStackId++;
		}
	});

	public static void logInfo(String message) {
		LOGGER.info(LOG_PREFIX + message);
	}

	public static void logWarn(String message) {
		LOGGER.warn(LOG_PREFIX + message);
	}

	public static void logError(String message) {
		LOGGER.error(LOG_PREFIX + message);
	}

	@SuppressWarnings("unused")
	public static boolean hasLastReadNbt() {
		return lastReadNbtPresent;
	}

	@SuppressWarnings("unused")
	public static void clearLastReadNbt() {
		lastReadNbt = null;
		lastReadNbtPresent = false;
	}

	public static void setLastReadNbt(class_2487 nbt) {
		lastReadNbt = nbt;
		lastReadNbtPresent = true;
	}

	public static class_2487 useLastReadNbt() {
		class_2487 result = null;
		if (lastReadNbt != null) {
			result = lastReadNbt.method_10553();
			lastReadNbt = null;
		}
		lastReadNbtPresent = false;
		return result;
	}

	@Override
	public void onInitialize() {
		ServerLoginConnectionEvents.QUERY_START.register((handler, server, sender, synchronizer) -> {
			sender.sendPacket(PRESENCE_CHANNEL, new class_2540(Unpooled.buffer()));
		});
		ServerLoginConnectionEvents.DISCONNECT.register((handler, server) -> {
			hasModClientConnectionHashes.remove(handler.method_2872().hashCode());
		});
		ServerLoginNetworking.registerGlobalReceiver(PRESENCE_CHANNEL, (server, handler, understood, buf, synchronizer, responseSender) -> {
			if (understood) {
				hasModClientConnectionHashes.add(handler.method_2872().hashCode());
			}
		});
		ServerPlayConnectionEvents.JOIN.register((handler, sender, server) -> {
			if (hasModClientConnectionHashes.contains(handler.method_2872().hashCode())) {
				((IServerPlayerEntity) handler.field_14140).nbtCrafting$setClientModPresent(true);
				hasModClientConnectionHashes.remove(handler.method_2872().hashCode());
			}
		});
	}

	public static boolean hasClientMod(class_3222 playerEntity) {
		if (playerEntity instanceof IServerPlayerEntity) {
			return ((IServerPlayerEntity) playerEntity).nbtCrafting$hasClientMod();
		}
		return false;
	}

	public static <T extends class_1860<?>> class_3956<T> registerRecipeType(String name) {
		class_2960 recipeTypeId = new class_2960(MOD_ID, name);
		RecipeTypeHelper.addToSyncBlacklist(recipeTypeId);
		return class_2378.method_10230(class_2378.field_17597, recipeTypeId, new class_3956<T>() {
			@Override
			public String toString() {
				return MOD_ID + ":" + name;
			}
		});
	}

	public static <S extends class_1865<T>, T extends class_1860<?>> S registerRecipeSerializer(String name, S recipeSerializer) {
		class_2960 serializerId = new class_2960(MOD_ID, name);
		RecipeTypeHelper.addToSyncBlacklist(serializerId);
		return class_2378.method_10230(class_2378.field_17598, serializerId, recipeSerializer);
	}

	public static List<class_2540> createAdvancedRecipeSyncPackets(class_1863 recipeManager) {
		advancedIngredientSerializationEnabled.set(true);
		List<class_1860<?>> recipes = recipeManager.method_8126().stream().filter(recipe -> {
			for (class_1856 ingredient : recipe.method_8117()) {
				if (((IIngredient) (Object) ingredient).nbtCrafting$isAdvanced()) {
					return true;
				}
			}
			return false;
		}).collect(Collectors.toList());

		List<class_2540> packets = new ArrayList<>();
		class_2540 buf = new class_2540(Unpooled.buffer());
		buf.method_10804(0);

		for (class_1860<?> recipe : recipes) {
			@SuppressWarnings("rawtypes")
			class_1865 serializer = recipe.method_8119();
			buf.method_10812(class_2378.field_17598.method_10221(serializer));
			buf.method_10812(recipe.method_8114());
			//noinspection unchecked
			serializer.method_8124(buf, recipe);

			if (buf.readableBytes() > 1_900_000) { // max packet size is 2^21=2_097_152 bytes
				packets.add(buf);
				buf = new class_2540(Unpooled.buffer());
				buf.method_10804(0);
			}
		}
		advancedIngredientSerializationEnabled.set(false);
		return packets;
	}

	public static boolean isAdvancedIngredientSerializationEnabled() {
		return advancedIngredientSerializationEnabled.get() != null
				&& advancedIngredientSerializationEnabled.get();
	}
}
