/*
 * Copyright 2020-2022 Siphalor
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 */

package de.siphalor.nbtcrafting3.api.nbt;

import java.util.List;
import java.util.Map;
import java.util.StringJoiner;

import com.google.gson.JsonElement;
import com.mojang.serialization.Dynamic;
import net.minecraft.class_1799;
import net.minecraft.class_2481;
import net.minecraft.class_2483;
import net.minecraft.class_2487;
import net.minecraft.class_2489;
import net.minecraft.class_2494;
import net.minecraft.class_2497;
import net.minecraft.class_2499;
import net.minecraft.class_2503;
import net.minecraft.class_2509;
import net.minecraft.class_2514;
import net.minecraft.class_2516;
import net.minecraft.class_2519;
import net.minecraft.class_2520;
import net.minecraft.nbt.*;
import org.apache.commons.lang3.ArrayUtils;
import org.jetbrains.annotations.Contract;

import de.siphalor.nbtcrafting3.NbtCrafting;
import de.siphalor.nbtcrafting3.dollar.DollarUtil;
import de.siphalor.nbtcrafting3.util.BetterJsonOps;

@SuppressWarnings("unused")
public class NbtUtil {
	public static final class_2487 EMPTY_COMPOUND = new class_2487();

	public static class_2487 getTagOrEmpty(class_1799 itemStack) {
		if (itemStack.method_7985())
			return itemStack.method_7969();
		else
			return EMPTY_COMPOUND;
	}

	public static class_2487 copyOrEmpty(class_2487 compoundTag) {
		if (compoundTag == null)
			return new class_2487();
		else
			return compoundTag.method_10553();
	}

	public static boolean tagsMatch(class_2520 main, class_2520 reference) {
		// Empty reference string is treated as wildcard
		if (isString(reference) && reference.method_10714().equals(""))
			return true;
		if (isString(main) && isString(reference))
			return main.method_10714().equals(reference.method_10714());
		if (isNumeric(main)) {
			if (isNumeric(reference))
				return asNumberTag(main).method_10697() == asNumberTag(reference).method_10697();
			// The reference might be a numeric range
			if (isString(reference) && reference.method_10714().startsWith("$"))
				return NbtNumberRange.ofString(reference.method_10714().substring(1)).matches(asNumberTag(main).method_10697());
			return false;
		}
		return false;
	}

	public static boolean compoundsOverlap(class_2487 main, class_2487 reference) {
		for (String key : main.method_10541()) {
			if (!reference.method_10545(key))
				continue;
			class_2520 mainTag = main.method_10580(key);
			class_2520 refTag = reference.method_10580(key);
			if (isCompound(mainTag) && isCompound(refTag)) {
				if (compoundsOverlap(main.method_10562(key), reference.method_10562(key)))
					return true;
			} else if (isList(mainTag) && isList(refTag)) {
				// noinspection ConstantConditions
				if (listsOverlap(asListTag(main.method_10580(key)), asListTag(reference.method_10580(key))))
					return true;
			} else if (tagsMatch(main.method_10580(key), reference.method_10580(key))) {
				return true;
			}
		}
		return false;
	}

	public static boolean listsOverlap(class_2483<class_2520> main, class_2483<class_2520> reference) {
		for (class_2520 mainTag : main) {
			for (class_2520 referenceTag : main) {
				if (isCompound(mainTag) && isCompound(referenceTag)) {
					if (compoundsOverlap(asCompoundTag(mainTag), asCompoundTag(referenceTag)))
						return true;
				} else if (isList(mainTag) && isList(referenceTag)) {
					if (listsOverlap(asListTag(mainTag), asListTag(referenceTag)))
						return true;
				} else if (tagsMatch(mainTag, referenceTag)) {
					return true;
				}
			}
		}
		return false;
	}

	public static boolean isCompoundContained(class_2487 inner, class_2487 outer) {
		for (String key : inner.method_10541()) {
			class_2520 innerTag = inner.method_10580(key);
			if (!outer.method_10545(key))
				return false;
			class_2520 outerTag = outer.method_10580(key);
			if (isCompound(innerTag) && isCompound(outerTag)) {
				if (isCompoundContained(asCompoundTag(innerTag), asCompoundTag(outerTag)))
					continue;
				return false;
			} else if (isList(innerTag) && isList(outerTag)) {
				if (isListContained(asListTag(innerTag), asListTag(outerTag)))
					continue;
				return false;
			} else if (tagsMatch(outerTag, innerTag))
				continue;
			return false;
		}
		return true;
	}

	public static boolean isListContained(class_2483<class_2520> inner, class_2483<class_2520> outer) {
		for (class_2520 innerTag : inner) {
			boolean success = false;
			for (class_2520 outerTag : outer) {
				if (isCompound(innerTag) && isCompound(outerTag) && isCompoundContained(asCompoundTag(innerTag), asCompoundTag(outerTag))) {
					success = true;
					break;
				} else if (isList(innerTag) && isList(outerTag) && isListContained(asListTag(innerTag), asListTag(outerTag))) {
					success = true;
					break;
				} else if (tagsMatch(innerTag, outerTag)) {
					success = true;
					break;
				}
			}
			if (!success)
				return false;
		}
		return true;
	}

	public static boolean sameType(class_2520 tag1, class_2520 tag2) {
		return tag1.method_10711() == tag2.method_10711();
	}

	@Contract(value = "null -> false", pure = true)
	public static boolean isString(class_2520 tag) {
		return tag instanceof class_2519;
	}

	@Contract(value = "null -> false", pure = true)
	public static boolean isCompound(class_2520 tag) {
		return tag instanceof class_2487;
	}

	@Contract(value = "null -> false", pure = true)
	public static boolean isList(class_2520 tag) {
		return tag instanceof class_2483;
	}

	@Contract(value = "null -> false", pure = true)
	public static boolean isNumeric(class_2520 tag) {
		return tag instanceof class_2514;
	}

	public static String asString(class_2520 tag) {
		if (tag instanceof class_2514) {
			return ((class_2514) tag).method_10702().toString();
		} else if (tag instanceof class_2519) {
			return tag.method_10714();
		} else if (tag instanceof class_2499) {
			StringJoiner joiner = new StringJoiner(", ");
			for (class_2520 entry : ((class_2499) tag)) {
				String s = asString(entry);
				joiner.add(s);
			}
			return joiner.toString();
		} else {
			return tag.toString();
		}
	}

	public static class_2519 asStringTag(class_2520 tag) {
		return (class_2519) tag;
	}

	public static class_2487 asCompoundTag(class_2520 tag) {
		return (class_2487) tag;
	}

	public static class_2483<class_2520> asListTag(class_2520 tag) {
		//noinspection unchecked
		return (class_2483<class_2520>) tag;
	}

	public static class_2514 asNumberTag(class_2520 tag) {
		return (class_2514) tag;
	}

	public static class_2520 getTag(class_2520 main, String path) {
		return getTag(main, splitPath(path));
	}

	public static class_2520 getTag(class_2520 main, String[] pathKeys) {
		class_2520 currentTag = main;
		for (String pathKey : pathKeys) {
			if ("".equals(pathKey))
				continue;
			if (currentTag == null)
				return null;
			if (pathKey.charAt(0) == '[') {
				int index = Integer.parseUnsignedInt(pathKey.substring(1, pathKey.length() - 2), 10);
				if (isList(currentTag)) {
					class_2483<class_2520> list = asListTag(currentTag);
					if (index >= list.size())
						return null;
					else
						currentTag = list.get(index);
				} else {
					return null;
				}
			} else {
				if (isCompound(currentTag)) {
					class_2487 compound = asCompoundTag(currentTag);
					if (compound.method_10545(pathKey)) {
						currentTag = compound.method_10580(pathKey);
					} else {
						return null;
					}
				} else {
					return null;
				}
			}
		}
		return currentTag;
	}

	public static class_2520 getTagOrCreate(class_2520 main, String path) throws NbtException {
		return getTagOrCreate(main, splitPath(path));
	}

	public static class_2520 getTagOrCreate(class_2520 main, String[] pathParts) throws NbtException {
		class_2520 currentTag = main;
		for (String pathPart : pathParts) {
			if ("".equals(pathPart))
				continue;
			if (pathPart.charAt(0) == '[') {
				if (!isList(currentTag)) {
					throw new NbtException(String.join(".", pathParts) + " doesn't match on " + main.method_10714());
				}
				class_2483<class_2520> currentList = asListTag(currentTag);
				int index = Integer.parseUnsignedInt(pathPart.substring(1, pathPart.length() - 1));
				if (currentList.size() <= index) {
					throw new NbtException(String.join(".", pathParts) + " contains invalid list in " + main.method_10714());
				} else if (isCompound(currentList.get(index)) || isList(currentList.get(index))) {
					currentTag = currentList.get(index);
				} else {
					throw new NbtException(String.join(".", pathParts) + " doesn't match on " + main.method_10714());
				}
			} else {
				if (!isCompound(currentTag)) {
					throw new NbtException(String.join(".", pathParts) + " doesn't match on " + main.method_10714());
				}
				class_2487 currentCompound = asCompoundTag(currentTag);
				if (!currentCompound.method_10545(pathPart)) {
					class_2487 newCompound = new class_2487();
					currentCompound.method_10566(pathPart, newCompound);
					currentTag = newCompound;
				} else if (isCompound(currentCompound.method_10580(pathPart)) || isList(currentCompound.method_10580(pathPart))) {
					currentTag = currentCompound.method_10580(pathPart);
				} else {
					throw new NbtException(String.join(".", pathParts) + " doesn't match on " + main.method_10714());
				}
			}
		}
		return currentTag;
	}

	public static void put(class_2520 main, String[] pathParts, class_2520 tag) throws NbtException {
		class_2520 parent = getTagOrCreate(main, ArrayUtils.subarray(pathParts, 0, pathParts.length - 1));

		String key = pathParts[pathParts.length - 1];
		if (key.charAt(0) == '[') {
			int i = Integer.parseUnsignedInt(key.substring(1, key.length() - 1));

			if (isList(parent)) {
				if (tag == null) {
					asListTag(parent).method_10536(i);
				} else {
					try {
						asListTag(parent).method_10531(i, tag);
					} catch (Exception e) {
						throw new NbtException("Can't add tag " + tag.method_10714() + " to list: " + parent.method_10714());
					}
				}
			} else {
				throw new NbtException(String.join(".", pathParts) + " doesn't match on " + main.method_10714());
			}
		} else {
			if (isCompound(parent)) {
				if (tag == null) {
					asCompoundTag(parent).method_10551(key);
				} else {
					asCompoundTag(parent).method_10566(key, tag);
				}
			} else {
				throw new NbtException(String.join(".", pathParts) + " doesn't match on " + main.method_10714());
			}
		}
	}

	public static String[] splitPath(String path) {
		return path.split("\\.|(?=\\[)");
	}

	public static String getLastKey(String path) {
		return path.substring(path.lastIndexOf('.') + 1);
	}

	public static void mergeInto(class_2487 target, class_2487 additions, boolean replace) {
		if (additions == null) return;

		for (String key : additions.method_10541()) {
			if (!target.method_10545(key)) {
				//noinspection ConstantConditions
				target.method_10566(key, additions.method_10580(key).method_10707());
				continue;
			}

			class_2520 targetTag = target.method_10580(key);
			class_2520 additionsTag = additions.method_10580(key);
			if (isCompound(targetTag) && isCompound(additionsTag)) {
				mergeInto(asCompoundTag(targetTag), asCompoundTag(additionsTag), replace);
			} else if (isList(targetTag) && isList(additionsTag)) {
				int targetSize = asListTag(targetTag).size();
				class_2483<class_2520> listTag = asListTag(targetTag);
				for (class_2520 tag : asListTag(additionsTag)) {
					class_2520 copy = tag.method_10707();
					listTag.add(tag);
				}
			} else {
				if (replace) {
					//noinspection ConstantConditions
					target.method_10566(key, additionsTag.method_10707());
				}
			}
		}
	}


	public static void mergeInto(class_2487 target, class_2487 additions, MergeContext context, String basePath) {
		if (additions == null) return;

		if (!basePath.isEmpty()) basePath += '.';

		for (String key : additions.method_10541()) {
			String path = basePath + key;
			class_2520 targetTag = target.method_10580(key);

			MergeBehavior mergeBehavior = context.getMergeMode(targetTag, path);

			class_2520 additionTag = additions.method_10580(key);
			assert additionTag != null;
			class_2520 merged = mergeBehavior.merge(targetTag, additionTag, context, path);
			if (merged != targetTag) {
				target.method_10566(key, merged);
			}
		}
	}

	public static void mergeInto(class_2483<class_2520> target, class_2483<class_2520> additions, MergeContext context, String basePath) {
		if (additions == null) return;

		int targetSize = target.size();
		int additionsSize = additions.size();

		for (int i = 0; i < additions.size() && i < targetSize; i++) { // for all elements that exist in both
			String path = basePath + "[" + i + "]";
			class_2520 targetTag = target.get(i);

			MergeBehavior mergeBehavior = context.getMergeMode(targetTag, path);

			if (mergeBehavior == MergeBehavior.APPEND) { // Legacy behavior
				try {
					target.add(additions.get(i).method_10707());
				} catch (Exception e) {
					NbtCrafting.logError("Can't append tag " + additions.get(i).method_10714() + " to list: " + target.method_10714());
				}
				continue;
			}

			class_2520 additionTag = additions.get(i);
			assert additionTag != null;
			class_2520 merged = mergeBehavior.merge(targetTag, additionTag, context, path);
			if (merged != targetTag) {
				target.method_10606(i, merged);
			}
		}

		for (int i = targetSize; i < additionsSize; i++) { // for any additional elements
			String path = basePath + "[" + i + "]";
			MergeBehavior mergeBehavior = context.getMergeMode(null, path);
			class_2520 tag = mergeBehavior.merge(null, additions.get(i), context, path);
			if (tag != null) {
				target.add(tag);
			}
		}
	}

	public static class_2520 asTag(Object value) {
		if (value instanceof class_2520) {
			return (class_2520) value;
		} else if (value instanceof String) {
			return class_2519.method_23256((String) value);
		} else if (value instanceof Float) {
			return class_2494.method_23244((Float) value);
		} else if (value instanceof Double) {
			return class_2489.method_23241((Double) value);
		} else if (value instanceof Byte) {
			return class_2481.method_23233((Byte) value);
		} else if (value instanceof Character) {
			return class_2519.method_23256(String.valueOf(value));
		} else if (value instanceof Short) {
			return class_2516.method_23254((Short) value);
		} else if (value instanceof Integer) {
			return class_2497.method_23247((Integer) value);
		} else if (value instanceof Long) {
			return class_2503.method_23251((Long) value);
		} else if (value instanceof Boolean) {
			return class_2481.method_23233((byte) ((Boolean) value ? 1 : 0));
		} else if (value instanceof List) {
			class_2499 listTag = new class_2499();
			for (Object element : (List<?>) value) {
				listTag.add(asTag(element));
			}
			return listTag;
		} else if (value instanceof Map) {
			class_2487 compoundTag = new class_2487();
			for (Map.Entry<?, ?> entry : ((Map<?, ?>) value).entrySet()) {
				compoundTag.method_10566(DollarUtil.asString(entry.getKey()), asTag(entry.getValue()));
			}
			return compoundTag;
		} else if (value instanceof class_1799) {
			return getTagOrEmpty(((class_1799) value));
		} else {
			return null;
		}
	}

	@SuppressWarnings("ConstantConditions")
	public static Object toDollarValue(class_2520 value) {
		if (value instanceof class_2519) {
			return value.method_10714();
		} else if (value instanceof class_2494) {
			return ((class_2494) value).method_10700();
		} else if (value instanceof class_2489) {
			return ((class_2489) value).method_10697();
		} else if (value instanceof class_2481) {
			return ((class_2481) value).method_10698();
		} else if (value instanceof class_2516) {
			return ((class_2516) value).method_10696();
		} else if (value instanceof class_2497) {
			return ((class_2497) value).method_10701();
		} else if (value instanceof class_2503) {
			return ((class_2503) value).method_10699();
		} else if (value instanceof class_2520) {
			return value;
		} else {
			return null;
		}
	}

	public static class_2520 asTag(JsonElement jsonElement) {
		return Dynamic.convert(BetterJsonOps.INSTANCE, class_2509.field_11560, jsonElement);
	}

	public static JsonElement toJson(class_2520 tag) {
		return Dynamic.convert(class_2509.field_11560, BetterJsonOps.INSTANCE, tag);
	}
}
