/*
 * Copyright 2020-2022 Siphalor
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 */

package de.siphalor.nbtcrafting3.mixin;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import com.google.gson.*;
import com.mojang.brigadier.StringReader;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import net.fabricmc.api.EnvType;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1856;
import net.minecraft.class_1869;
import net.minecraft.class_2378;
import net.minecraft.class_2487;
import net.minecraft.class_2522;
import net.minecraft.class_2540;
import net.minecraft.class_2960;
import net.minecraft.class_3489;
import net.minecraft.class_3494;
import net.minecraft.class_3518;
import org.spongepowered.asm.mixin.Mixin;
import org.spongepowered.asm.mixin.Shadow;
import org.spongepowered.asm.mixin.Unique;
import org.spongepowered.asm.mixin.injection.At;
import org.spongepowered.asm.mixin.injection.Inject;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfo;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfoReturnable;

import de.siphalor.nbtcrafting3.NbtCrafting;
import de.siphalor.nbtcrafting3.api.JsonPreprocessor;
import de.siphalor.nbtcrafting3.api.nbt.NbtUtil;
import de.siphalor.nbtcrafting3.dollar.reference.ReferenceResolver;
import de.siphalor.nbtcrafting3.ingredient.*;
import de.siphalor.nbtcrafting3.util.duck.ICloneable;

@Mixin(value = class_1856.class, priority = 990) // Inject before NbtC v2
public abstract class MixinIngredient implements IIngredient, ICloneable {
	@Shadow
	private class_1799[] matchingStacks;

	@Unique
	private IngredientEntry[] advancedEntries;

	@Override
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}

	@Inject(method = "<init>", at = @At("RETURN"))
	private void onConstruct(@SuppressWarnings("rawtypes") Stream stream, CallbackInfo ci) {
		advancedEntries = null;
	}

	@Inject(method = "cacheMatchingStacks", at = @At("HEAD"), cancellable = true)
	private void createStackArray(CallbackInfo callbackInfo) {
		if (advancedEntries != null) {
			callbackInfo.cancel();
			if (matchingStacks == null || matchingStacks.length == 0) {
				if (FabricLoader.getInstance().getEnvironmentType() == EnvType.CLIENT) {
					matchingStacks = Arrays.stream(advancedEntries).flatMap(entry -> entry.getPreviewStacks(true).stream()).distinct().toArray(class_1799[]::new);
				} else {
					matchingStacks = Arrays.stream(advancedEntries).flatMap(entry -> entry.getPreviewStacks(false).stream()).distinct().toArray(class_1799[]::new);
				}
				if (matchingStacks.length == 0) {
					matchingStacks = new class_1799[]{class_1799.field_8037};
				}
			}
		}
	}

	@Inject(method = "test", at = @At("HEAD"), cancellable = true)
	public void matches(class_1799 stack, CallbackInfoReturnable<Boolean> callbackInfoReturnable) {
		if (stack == null) {
			callbackInfoReturnable.setReturnValue(false);
			return;
		}
		if (advancedEntries != null) {
			if (advancedEntries.length == 0) {
				callbackInfoReturnable.setReturnValue(stack.method_7960());
				return;
			}
			for (IngredientEntry advancedEntry : advancedEntries) {
				if (advancedEntry.matches(stack)) {
					callbackInfoReturnable.setReturnValue(true);
					return;
				}
			}
			callbackInfoReturnable.setReturnValue(false);
		}
	}

	@Inject(method = "write", at = @At("HEAD"), cancellable = true)
	public void write(class_2540 buf, CallbackInfo callbackInfo) {
		if (NbtCrafting.isAdvancedIngredientSerializationEnabled()) {
			if (advancedEntries != null && advancedEntries.length != 0) {
				buf.method_10804(advancedEntries.length);
				for (IngredientEntry entry : advancedEntries) {
					buf.writeBoolean(entry instanceof IngredientMultiStackEntry);
					entry.write(buf);
				}
				callbackInfo.cancel();
			} else {
				// -1 is used to keep network compatibility with lower versions of Nbt Crafting,
				// that used 0 to just indicate no advanced ingredients
				buf.method_10804(-1);
			}
		}
	}

	@Inject(method = "toJson", at = @At("HEAD"), cancellable = true)
	public void toJson(CallbackInfoReturnable<JsonElement> callbackInfoReturnable) {
		if (advancedEntries != null) {
			if (advancedEntries.length == 1) {
				callbackInfoReturnable.setReturnValue(advancedEntries[0].toJson());
				return;
			}
			JsonArray array = new JsonArray();
			for (IngredientEntry advancedEntry : advancedEntries) {
				array.add(advancedEntry.toJson());
			}
			callbackInfoReturnable.setReturnValue(array);
		}
	}

	@Inject(method = "isEmpty", at = @At("HEAD"), cancellable = true)
	public void isEmpty(CallbackInfoReturnable<Boolean> callbackInfoReturnable) {
		if (advancedEntries != null) {
			callbackInfoReturnable.setReturnValue(advancedEntries.length == 0);
		}
	}

	@Unique
	private static class_1856 ofAdvancedEntries(Stream<? extends IngredientEntry> entries) {
		if (entries == null)
			NbtCrafting.logError("Internal error: can't construct ingredient from null entry stream!");
		try {
			class_1856 ingredient;
			//noinspection ConstantConditions
			ingredient = (class_1856) ((ICloneable) (Object) class_1856.field_9017).clone();
			((IIngredient) (Object) ingredient).nbtCrafting$setAdvancedEntries(entries);
			return ingredient;
		} catch (CloneNotSupportedException e) {
			e.printStackTrace();
		}
		return class_1856.field_9017;
	}

 	/*
 	 This is client side :/
	@Inject(method = "ofStacks", at = @At("HEAD"), cancellable = true)
	private static void ofStacks(ItemStack[] arr, CallbackInfoReturnable<Ingredient> callbackInfoReturnable) {
		if(Arrays.stream(arr).anyMatch(stack -> stack.hasTag())) {
			callbackInfoReturnable.setReturnValue(ofAdvancedEntries(Arrays.stream(arr).map(stack -> new IngredientStackEntry(stack))));
		}
	}
	*/

	@Inject(method = "fromPacket", at = @At("HEAD"), cancellable = true)
	private static void fromPacket(class_2540 buf, CallbackInfoReturnable<class_1856> cir) {
		if (NbtCrafting.isAdvancedIngredientSerializationEnabled()) {
			int length = buf.method_10816();
			if (length >= 0) {
				ArrayList<IngredientEntry> entries = new ArrayList<>(length);
				for (int i = 0; i < length; i++) {
					if (buf.readBoolean()) {
						entries.add(IngredientMultiStackEntry.read(buf));
					} else {
						entries.add(IngredientStackEntry.read(buf));
					}
				}
				cir.setReturnValue(ofAdvancedEntries(entries.stream()));
			}
		}
	}

	@Inject(method = "fromJson", at = @At("HEAD"), cancellable = true)
	private static void fromJson(JsonElement element, CallbackInfoReturnable<class_1856> callbackInfoReturnable) {
		if (!NbtCrafting.isAdvancedIngredientSerializationEnabled()) {
			return;
		}

		if (element == null || element.isJsonNull()) {
			throw new JsonSyntaxException("Item cannot be null");
		}
		if (element.isJsonObject()) {
			if (element.getAsJsonObject().has("data") || element.getAsJsonObject().has("remainder") || element.getAsJsonObject().has("potion"))
				callbackInfoReturnable.setReturnValue(ofAdvancedEntries(Stream.of(advancedEntryFromJson(element.getAsJsonObject()))));
		} else if (element.isJsonArray()) {
			final JsonArray jsonArray = element.getAsJsonArray();

			boolean containsCustomData = false;
			for (JsonElement jsonElement : jsonArray) {
				if (jsonElement.isJsonObject()) {
					JsonObject jsonObject = jsonElement.getAsJsonObject();
					if (jsonObject.has("data") || jsonObject.has("remainder") || jsonObject.has("potion")) {
						containsCustomData = true;
						break;
					}
				}
			}

			if (containsCustomData) {
				if (jsonArray.size() == 0) {
					throw new JsonSyntaxException("Item array cannot be empty, at least one item must be defined");
				}
				callbackInfoReturnable.setReturnValue(ofAdvancedEntries(StreamSupport.stream(jsonArray.spliterator(), false).map(e -> advancedEntryFromJson(class_3518.method_15295(e, "item")))));
			}
		}
	}

	@Unique
	private static IngredientEntry advancedEntryFromJson(JsonObject jsonObject) {
		if (jsonObject.has("item") && jsonObject.has("tag")) {
			throw new JsonParseException("An ingredient entry is either a tag or an item or a potion, not both");
		}
		if (jsonObject.has("item")) {
			final class_2960 identifier = new class_2960(class_3518.method_15265(jsonObject, "item"));
			try {
				final class_1792 item = class_2378.field_11142.method_17966(identifier).orElseThrow(() -> {
					throw new JsonSyntaxException("Unknown item '" + identifier.toString() + "'");
				});
				IngredientStackEntry entry = new IngredientStackEntry(class_2378.field_11142.method_10206(item), loadIngredientEntryCondition(jsonObject));
				if (jsonObject.has("remainder")) {
					entry.setRecipeRemainder(class_1869.method_8155(class_3518.method_15296(jsonObject, "remainder")));
				}
				return entry;
			} catch (Throwable e) {
				e.printStackTrace();
				return null;
			}
		}
		if (jsonObject.has("potion")) {
			final class_2960 identifier = new class_2960(class_3518.method_15265(jsonObject, "potion"));
			try {
				class_2378.field_11143.method_17966(identifier).orElseThrow(() -> {
					throw new JsonSyntaxException("Unknown potion '" + identifier.toString() + "'");
				});
				IngredientEntryCondition condition = loadIngredientEntryCondition(jsonObject);
				if (condition.requiredElements == NbtUtil.EMPTY_COMPOUND)
					condition.requiredElements = new class_2487();
				condition.requiredElements.method_10582("Potion", identifier.toString());
				IngredientStackEntry entry = new IngredientStackEntry(class_2378.field_11142.method_10206(class_1802.field_8574), condition);
				if (jsonObject.has("remainder")) {
					entry.setRecipeRemainder(class_1869.method_8155(class_3518.method_15296(jsonObject, "remainder")));
				}
				return entry;
			} catch (Throwable e) {
				e.printStackTrace();
				return null;
			}
		}
		if (!jsonObject.has("tag")) {
			throw new JsonParseException("An ingredient entry needs either a tag or an item");
		}
		final class_2960 identifier2 = new class_2960(class_3518.method_15265(jsonObject, "tag"));
		final class_3494<class_1792> tag = class_3489.method_15106().method_30210(identifier2);
		if (tag == null) {
			throw new JsonSyntaxException("Unknown item tag '" + identifier2 + "'");
		}
		IngredientMultiStackEntry entry = new IngredientMultiStackEntry(tag.method_15138().stream().map(class_2378.field_11142::method_10206).collect(Collectors.toList()), loadIngredientEntryCondition(jsonObject));
		entry.setTag(identifier2.toString());
		if (jsonObject.has("remainder")) {
			entry.setRecipeRemainder(class_1869.method_8155(class_3518.method_15296(jsonObject, "remainder")));
		}
		return entry;
	}

	@Unique
	private static IngredientEntryCondition loadIngredientEntryCondition(JsonObject jsonObject) {
		if (jsonObject.has("data")) {
			if (class_3518.method_15289(jsonObject, "data")) {
				try {
					class_2487 compoundTag = new class_2522(new StringReader(jsonObject.get("data").getAsString())).method_10727();
					return new IngredientEntryCondition(compoundTag, NbtUtil.EMPTY_COMPOUND);
				} catch (CommandSyntaxException e) {
					e.printStackTrace();
				}
			} else if (jsonObject.get("data").isJsonObject()) {
				return IngredientEntryCondition.fromJson((JsonObject) JsonPreprocessor.process(jsonObject.get("data").getAsJsonObject()));
			}
		}
		return IngredientEntryCondition.EMPTY;
	}

	@Override
	public boolean nbtCrafting3$isAdvanced() {
		return advancedEntries != null;
	}

	@Override
	public void nbtCrafting$setAdvancedEntries(Stream<? extends IngredientEntry> entries) {
		advancedEntries = entries.filter(Objects::nonNull).toArray(IngredientEntry[]::new);
	}

	@Override
	public class_1799 nbtCrafting3$getRecipeRemainder(class_1799 stack, ReferenceResolver referenceResolver) {
		if (advancedEntries != null) {
			for (IngredientEntry entry : advancedEntries) {
				if (entry.matches(stack)) {
					class_1799 remainder = entry.getRecipeRemainder(stack, referenceResolver);
					if (remainder != null) {
						return remainder;
					}
				}
			}
		}
		return null;
	}
}
