/*
 * Copyright 2020-2022 Siphalor
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 */

package de.siphalor.nbtcrafting3.ingredient;

import java.util.ArrayList;
import java.util.List;
import net.minecraft.class_1799;
import net.minecraft.class_2378;
import net.minecraft.class_2487;
import net.minecraft.class_2520;
import net.minecraft.class_2540;
import net.minecraft.class_2960;
import net.minecraft.class_3518;
import net.minecraft.class_3545;
import com.google.gson.*;
import de.siphalor.nbtcrafting3.NbtCrafting;
import de.siphalor.nbtcrafting3.api.nbt.NbtException;
import de.siphalor.nbtcrafting3.api.nbt.NbtIterator;
import de.siphalor.nbtcrafting3.api.nbt.NbtNumberRange;
import de.siphalor.nbtcrafting3.api.nbt.NbtUtil;
import de.siphalor.nbtcrafting3.dollar.DollarExtractor;
import de.siphalor.nbtcrafting3.dollar.DollarUtil;
import de.siphalor.nbtcrafting3.dollar.exception.DollarEvaluationException;
import de.siphalor.nbtcrafting3.dollar.exception.UnresolvedDollarReferenceException;
import de.siphalor.nbtcrafting3.dollar.part.DollarPart;

public class IngredientEntryCondition {
	public static final IngredientEntryCondition EMPTY = new IngredientEntryCondition(NbtUtil.EMPTY_COMPOUND, NbtUtil.EMPTY_COMPOUND);

	public class_2487 requiredElements;
	public class_2487 deniedElements;
	public List<class_3545<String, DollarPart>> dollarPredicates;
	private class_2487 previewTag;

	protected IngredientEntryCondition() {
		requiredElements = NbtUtil.EMPTY_COMPOUND;
		deniedElements = NbtUtil.EMPTY_COMPOUND;
		dollarPredicates = null;
	}

	public IngredientEntryCondition(class_2487 requiredElements, class_2487 deniedElements) {
		this.requiredElements = requiredElements;
		this.deniedElements = deniedElements;
	}

	public boolean matches(class_1799 stack) {
		if (!stack.method_7985()) {
			return requiredElements.method_33133();
		}
		class_2487 tag = stack.method_7969();
		//noinspection ConstantConditions
		if (!deniedElements.method_33133() && NbtUtil.compoundsOverlap(tag, deniedElements))
			return false;
		if (!requiredElements.method_33133() && !NbtUtil.isCompoundContained(requiredElements, tag))
			return false;
		if (dollarPredicates != null && !dollarPredicates.isEmpty()) {
			for (class_3545<String, DollarPart> predicate : dollarPredicates) {
				try {
					if (!DollarUtil.asBoolean(predicate.method_15441().evaluate(ref -> {
						if ("$".equals(ref)) {
							return tag;
						}
						throw new UnresolvedDollarReferenceException(ref);
					}))) {
						return false;
					}
				} catch (DollarEvaluationException e) {
					NbtCrafting.logWarn("Failed to evaluate dollar predicate (" + predicate.method_15442() + "): " + e.getMessage());
				}
			}
		}
		return true;
	}

	public void addToJson(JsonObject json) {
		if (requiredElements.method_10546() > 0)
			json.add("require", NbtUtil.toJson(requiredElements));
		if (deniedElements.method_10546() > 0)
			json.add("deny", NbtUtil.toJson(deniedElements));
		if (dollarPredicates != null && !dollarPredicates.isEmpty()) {
			JsonArray array = new JsonArray();
			for (class_3545<String, DollarPart> condition : dollarPredicates) {
				array.add(condition.method_15442());
			}
			json.add("conditions", array);
		}
	}

	public class_2487 getPreviewTag() {
		if (previewTag == null) {
			previewTag = requiredElements.method_10553();
			List<class_3545<String[], class_2520>> dollarRangeKeys = new ArrayList<>();
			NbtIterator.iterateTags(previewTag, (path, key, tag) -> {
				if (NbtUtil.isString(tag)) {
					String text = NbtUtil.asString(tag);
					if (text.startsWith("$")) {
						dollarRangeKeys.add(new class_3545<>(NbtUtil.splitPath(path + key), NbtNumberRange.ofString(text.substring(1)).getExample()));
					}
				}
				return NbtIterator.Action.RECURSE;
			});
			for (class_3545<String[], class_2520> dollarRangeKey : dollarRangeKeys) {
				try {
					NbtUtil.put(previewTag, dollarRangeKey.method_15442(), dollarRangeKey.method_15441());
				} catch (NbtException e) {
					NbtCrafting.logWarn("Failed to set dollar range value " + dollarRangeKey.method_15441() + " for key " + String.join(".", dollarRangeKey.method_15442()) + " in preview tag " + previewTag);
				}
			}
		}
		return previewTag;
	}

	public static IngredientEntryCondition fromJson(JsonObject json) {
		IngredientEntryCondition condition = new IngredientEntryCondition();

		boolean flatObject = true;

		if (json.has("require")) {
			if (!json.get("require").isJsonObject())
				throw new JsonParseException("data.require must be an object");
			condition.requiredElements = (class_2487) NbtUtil.asTag(json.getAsJsonObject("require"));
			flatObject = false;
		}
		if (json.has("potion")) {
			class_2960 potion = new class_2960(class_3518.method_15265(json, "potion"));
			if (class_2378.field_11143.method_17966(potion).isPresent()) {
				if (condition.requiredElements == NbtUtil.EMPTY_COMPOUND) {
					condition.requiredElements = new class_2487();
				}
				condition.requiredElements.method_10582("Potion", potion.toString());
			} else {
				new JsonSyntaxException("Unknown potion '" + potion + "'").printStackTrace();
			}
			flatObject = false;
		}
		if (json.has("potion")) {
			class_2960 potion = new class_2960(class_3518.method_15265(json, "potion"));
			if (class_2378.field_11143.method_17966(potion).isPresent()) {
				if (condition.requiredElements == NbtUtil.EMPTY_COMPOUND) {
					condition.requiredElements = new class_2487();
				}
				condition.requiredElements.method_10582("Potion", potion.toString());
			} else {
				new JsonSyntaxException("Unknown potion '" + potion + "'").printStackTrace();
			}
			flatObject = false;
		}
		if (json.has("deny")) {
			if (!json.get("deny").isJsonObject())
				throw new JsonParseException("data.deny must be an object");
			condition.deniedElements = (class_2487) NbtUtil.asTag(json.getAsJsonObject("deny"));
			flatObject = false;
		}
		if (json.has("conditions")) {
			if (!json.get("conditions").isJsonArray())
				throw new JsonParseException("data.conditions must be an array");
			JsonArray array = json.getAsJsonArray("conditions");
			List<class_3545<String, DollarPart>> predicates = new ArrayList<>(array.size());
			for (JsonElement jsonElement : array) {
				if (!class_3518.method_15286(jsonElement))
					throw new JsonParseException("data.conditions must be an array of strings");
				predicates.add(new class_3545<>(jsonElement.getAsString(), DollarExtractor.parse(jsonElement.getAsString())));
			}
			condition.dollarPredicates = predicates;
			flatObject = false;
		}

		if (flatObject) {
			condition.requiredElements = (class_2487) NbtUtil.asTag(json);
		}

		return condition;
	}

	public void write(class_2540 buf) {
		buf.method_10794(requiredElements);
		buf.method_10794(deniedElements);
	}

	public static IngredientEntryCondition read(class_2540 buf) {
		return new IngredientEntryCondition(buf.method_10798(), buf.method_10798());
	}
}
