/*
 * Copyright 2021 Siphalor
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 */

package de.siphalor.nmuk.impl.mixin;

import de.siphalor.nmuk.NMUK;
import de.siphalor.nmuk.impl.IKeyBinding;
import de.siphalor.nmuk.impl.NMUKKeyBindingHelper;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import org.apache.logging.log4j.Level;
import org.spongepowered.asm.mixin.*;
import org.spongepowered.asm.mixin.injection.At;
import org.spongepowered.asm.mixin.injection.Inject;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfo;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import net.minecraft.class_304;
import net.minecraft.class_310;
import net.minecraft.class_315;
import net.minecraft.class_3675;

@Mixin(value = class_315.class, priority = 800)
public class MixinGameOptions {
	@Unique
	private File nmukOptionsFile;
	@Unique
	private class_304[] tempKeysAll;

	@Mutable
	@Shadow
	@Final
	public class_304[] keysAll;

	// Prevent nmuk keybindings from getting saved to the Vanilla options file
	@Inject(
			method = "write",
			at = @At(value = "FIELD", target = "Lnet/minecraft/client/options/GameOptions;keysAll:[Lnet/minecraft/client/options/KeyBinding;")
	)
	public void removeNMUKBindings(CallbackInfo ci) {
		tempKeysAll = keysAll;
		keysAll = Arrays.stream(keysAll).filter(binding -> !((IKeyBinding) binding).nmuk_isAlternative()).toArray(class_304[]::new);
	}

	@Inject(
			method = "write",
			at = @At(value = "INVOKE", target = "Lnet/minecraft/sound/SoundCategory;values()[Lnet/minecraft/sound/SoundCategory;")
	)
	public void resetAllKeys(CallbackInfo ci) {
		keysAll = tempKeysAll;
	}

	@Inject(
			method = "write",
			at = @At("RETURN")
	)
	public void save(CallbackInfo ci) {
		try (PrintWriter printWriter = new PrintWriter(new OutputStreamWriter(new FileOutputStream(nmukOptionsFile), StandardCharsets.UTF_8))) {
			for (class_304 binding : keysAll) {
				if (((IKeyBinding) binding).nmuk_isAlternative()) {
					printWriter.println("key_" + binding.method_1431() + ":" + binding.method_1428());
				}
			}
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
	}

	@Inject(
			method = "load",
			at = @At("RETURN")
	)
	public void load(CallbackInfo ci) {
		if (nmukOptionsFile == null) {
			nmukOptionsFile = new File(class_310.method_1551().field_1697, "options." + NMUK.MOD_ID + ".txt");
		}

		if (!nmukOptionsFile.exists()) {
			return;
		}
		Map<String, class_304> keyBindings = KeyBindingAccessor.getKeysById();
		Object2IntMap<class_304> alternativeCountMap = new Object2IntOpenHashMap<>();
		Queue<class_304> newAlternatives = new ConcurrentLinkedQueue<>();
		try (BufferedReader reader = new BufferedReader(new FileReader(nmukOptionsFile))) {
			String line;
			while ((line = reader.readLine()) != null) {
				try {
					int stringIndex = line.lastIndexOf(':');
					if (stringIndex <= 0) {
						NMUK.log(Level.WARN, "Invalid nmuk options line: " + line);
						continue;
					}
					String id = line.substring(0, stringIndex);
					String keyId = line.substring(stringIndex + 1);
					if (!id.startsWith("key_")) {
						NMUK.log(Level.WARN, "Invalid nmuk options entry: " + id);
						continue;
					}
					id = id.substring(4);
					stringIndex = id.indexOf('%');
					if (stringIndex <= 0) {
						NMUK.log(Level.WARN, "Nmuk entry is missing an alternative id");
						continue;
					}
					short altId = Short.parseShort(id.substring(stringIndex + 1));
					id = id.substring(0, stringIndex);
					class_3675.class_306 boundKey = class_3675.method_15981(keyId);
					//noinspection ConstantConditions
					class_304 base = keyBindings.get(id);
					if (base != null) {
						int index = alternativeCountMap.getOrDefault(base, 0);
						List<class_304> children = ((IKeyBinding) base).nmuk_getAlternatives();
						((IKeyBinding) base).nmuk_setNextChildId(altId);
						if (children == null) {
							class_304 alternative = NMUKKeyBindingHelper.createAlternativeKeyBinding(base);
							alternative.method_1422(boundKey);
							newAlternatives.add(alternative);
						} else {
							if (index < children.size()) {
								children.get(index).method_1422(boundKey);
							} else {
								class_304 alternative = NMUKKeyBindingHelper.createAlternativeKeyBinding(base);
								alternative.method_1422(boundKey);
								newAlternatives.add(alternative);
							}
						}
						alternativeCountMap.putIfAbsent(base, index + 1);
					}
				} catch (Throwable e) {
					NMUK.log(Level.ERROR, "Encountered an issue whilst loading nmuk options file!");
					e.printStackTrace();
				}
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
		int newCount, oldCount;
		for (class_304 binding : keysAll) {
			newCount = alternativeCountMap.getOrDefault(binding, 0);
			oldCount = ((IKeyBinding) binding).nmuk_getAlternativesCount();
			if (oldCount > newCount) {
				List<class_304> alternatives = ((IKeyBinding) binding).nmuk_getAlternatives();
				alternatives.subList(newCount, oldCount).clear();
			}
		}
		NMUKKeyBindingHelper.registerKeyBindings((class_315) (Object) this, newAlternatives);
	}
}
