package de.siphalor.pushtocraft;

import com.google.common.collect.Streams;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import net.minecraft.class_1865;
import net.minecraft.class_2378;
import net.minecraft.class_2960;
import net.minecraft.class_3300;
import net.minecraft.class_3545;
import net.minecraft.class_3695;
import net.minecraft.class_4309;

public class PushToCraftManager extends class_4309 {
	private static final Gson GSON = (new GsonBuilder()).setPrettyPrinting().disableHtmlEscaping().create();

	private final Map<String, Collection<Entry>> exactPushes = new HashMap<>();
	private final Map<String, Collection<class_3545<Pattern, Entry>>> namespacePushes = new HashMap<>();
	private final Collection<class_3545<Pattern, Entry>> flexiblePushes = new LinkedList<>();
	private final Collection<Entry> wildcardPushes = new LinkedList<>();

	private static PushToCraftManager instance;

	public static PushToCraftManager getInstance() {
		return instance;
	}

	public PushToCraftManager() {
		super(GSON, "push_to_craft");
		instance = this;
	}

	public String method_22322() {
		return PushToCraft.MOD_ID;
	}

	@Override
	protected void apply(Map<class_2960, JsonElement> resources, class_3300 manager, class_3695 profiler) {
		exactPushes.clear();
		namespacePushes.clear();
		flexiblePushes.clear();
		wildcardPushes.clear();
		for (Map.Entry<class_2960, JsonElement> e : resources.entrySet()) {
			class_2960 identifier = e.getKey();
			if (!e.getValue().isJsonObject()) {
				logError(identifier, "must be a JSON Object");
				continue;
			}
			JsonObject jsonObject = e.getValue().getAsJsonObject();
			if (jsonObject.has("additions")) {
				JsonElement element = jsonObject.get("additions");
				if (element.isJsonArray()) {
					LinkedList<String> additions = new LinkedList<>();
					for (JsonElement additionElement : element.getAsJsonArray()) {
						if (Util.isString(additionElement)) {
							additions.add(additionElement.getAsString());
						} else {
							logWarn(identifier, "contains an illegal additions entry of type " + additionElement.getClass().getSimpleName());
						}
					}

					Collection<String> targets = null;
					if (jsonObject.has("targets")) {
						element = jsonObject.get("targets");
						if (element.isJsonArray()) {
							targets = new LinkedList<>();
							for (JsonElement targetElement : element.getAsJsonArray()) {
								if (Util.isString(targetElement)) {
									targets.add(targetElement.getAsString());
								} else {
									logWarn(identifier, "has item/tag target in array of invalid type " + targetElement.getClass().getSimpleName() + " - should be string");
								}
							}
						} else if (Util.isString(element)) {
							targets = Collections.singleton(element.getAsString());
						} else {
							logError(identifier, "has item/tag target of invalid type " + element.getClass().getSimpleName() + " - should be string or array of strings");
						}
					}
					if (targets == null || targets.isEmpty()) {
						logError(identifier, "targets no items or tags");
						continue;
					}

					if (jsonObject.has("recipes")) {
						element = jsonObject.get("recipes");
						if (element.isJsonObject()) {
							jsonObject = element.getAsJsonObject();
							Collection<class_1865<?>> recipeSerializers = Collections.emptyList();

							if (jsonObject.has("types")) {
								element = jsonObject.get("types");
								if (element.isJsonArray()) {
									recipeSerializers = new LinkedList<>();
									for (JsonElement recipeSerializerElement : element.getAsJsonArray()) {
										if (Util.isString(recipeSerializerElement)) {
											class_2960 recipeSerializerId = new class_2960(recipeSerializerElement.getAsString());
											if (class_2378.field_17598.method_10250(recipeSerializerId)) {
												recipeSerializers.add(class_2378.field_17598.method_10223(recipeSerializerId));
											} else {
												logWarn(identifier, "has recipe type specifier \"" + recipeSerializerId + "\" that could not be resolved to a recipe type");
											}
										} else {
											logWarn(identifier, "has recipe type specifier in array of invalid type " + recipeSerializerElement.getClass().getSimpleName() + " - should be a string");
										}
									}
									if (recipeSerializers.isEmpty()) {
										logError(identifier, "has empty or completely invalid recipe type specifier array");
										continue;
									}
								} else {
									logError(identifier, "has recipe type specifier of invalid type " + element.getClass().getSimpleName() + " - should be an array");
									continue;
								}
							}

							Entry entry = new Entry(targets, additions, recipeSerializers);

							if (jsonObject.has("ids")) {
								element = jsonObject.get("ids");
								if (element.isJsonArray()) {
									boolean valid = false;
									for (JsonElement idElement : element.getAsJsonArray()) {
										if (Util.isString(idElement)) {
											String id = idElement.getAsString();
											if (id.charAt(0) == '/' && id.endsWith("/")) {
												flexiblePushes.add(new class_3545<>(
														Pattern.compile(id.substring(1, id.length() - 1)),
														entry
												));
											} else {
												int index = id.indexOf(':');
												String namespace, path;
												if (index == -1) {
													namespace = "minecraft";
													path = id;
												} else {
													namespace = id.substring(0, index);
													path = id.substring(index + 1);
												}
												if (path.charAt(0) == '/' && path.endsWith("/")) {
													namespacePushes.computeIfAbsent(namespace, s -> new LinkedList<>())
															.add(new class_3545<>(Pattern.compile(path.substring(1, path.length() - 1)), entry));
												} else {
													exactPushes.computeIfAbsent(path, s -> new LinkedList<>())
															.add(entry);
												}
											}
											valid = true;
										} else {
											logWarn(identifier, "has recipe id in array of invalid type " + idElement.getClass().getSimpleName() + " - should be a string");
										}
									}
									if (!valid) {
										logError(identifier, "has no valid recipes ids in array");
									}
								} else {
									logError(identifier, "has recipe ids tag of invalid type " + element.getClass().getSimpleName() + " - should be an array");
								}
							} else {
								wildcardPushes.add(entry);
							}
						} else {
							logError(identifier, "has recipes specifier of invalid type " + element.getClass().getSimpleName() + " - should be an object");
						}
					} else {
						logError(identifier, "targets no recipes");
					}
				} else {
					logError(identifier, "has an ill-formed additions tag of type " + element.getClass().getSimpleName());
				}
			} else {
				logError(identifier, "is missing additions list. Sorry but pushing nothing makes no sense O_O");
			}
		}
	}

	private void logError(class_2960 identifier, String errorText) {
		PushToCraft.LOGGER.error("PushToCraft entry " + identifier + " " + errorText);
	}

	private void logWarn(class_2960 identifier, String warnText) {
		PushToCraft.LOGGER.warn("PushToCraft entry " + identifier + " " + warnText);
	}

	public Stream<Entry> getMatches(class_2960 recipeId, class_1865<?> recipeSerializer) {
		String recipeIdString = recipeId.toString();
		String path = recipeId.method_12832();
		//noinspection UnstableApiUsage
		return Streams.concat(
				exactPushes.getOrDefault(recipeIdString, Collections.emptyList()).parallelStream(),
				namespacePushes.getOrDefault(recipeId.method_12836(), Collections.emptyList()).parallelStream().filter(pair -> pair.method_15442().matcher(path).matches()).map(class_3545::method_15441),
				flexiblePushes.parallelStream().filter(pair -> pair.method_15442().matcher(recipeIdString).matches()).map(class_3545::method_15441),
				wildcardPushes.parallelStream()
		).filter(entry -> entry.recipeSerializerMatches(recipeSerializer));
	}

	public static class Entry {
		private final Collection<String> targets;
		private final Collection<String> additions;
		private final Collection<class_1865<?>> recipeSerializers;

		public Entry(Collection<String> targets, Collection<String> additions, Collection<class_1865<?>> recipeSerializers) {
			this.targets = targets;
			this.additions = additions;
			this.recipeSerializers = recipeSerializers;
		}

		public boolean recipeSerializerMatches(class_1865<?> recipeSerializer) {
			if (recipeSerializers.isEmpty())
				return true;
			return recipeSerializers.contains(recipeSerializer);
		}

		public Collection<String> getAdditions() {
			return additions;
		}

		public Collection<String> getTargets() {
			return targets;
		}
	}
}
