package de.siphalor.spiceoffabric.foodhistory;

import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import de.siphalor.spiceoffabric.SpiceOfFabric;
import de.siphalor.spiceoffabric.config.SOFConfig;
import de.siphalor.spiceoffabric.networking.SOFCommonNetworking;
import de.siphalor.spiceoffabric.util.IHungerManager;
import de.siphalor.spiceoffabric.util.queue.ArrayFixedLengthIntFIFOQueue;
import de.siphalor.spiceoffabric.util.queue.FixedLengthIntFIFOQueueWithStats;
import it.unimi.dsi.fastutil.ints.Int2IntArrayMap;
import it.unimi.dsi.fastutil.ints.Int2IntMap;
import it.unimi.dsi.fastutil.ints.IntIterator;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import net.minecraft.class_1324;
import net.minecraft.class_1657;
import net.minecraft.class_1702;
import net.minecraft.class_1799;
import net.minecraft.class_2487;
import net.minecraft.class_2497;
import net.minecraft.class_2499;
import net.minecraft.class_2520;
import net.minecraft.class_2540;
import net.minecraft.class_3222;
import net.minecraft.class_3532;
import net.minecraft.class_5134;

public class FoodHistory {

	protected static final String DICTIONARY_NBT_KEY = "dictionary";
	protected static final String RECENT_HISTORY_NBT_KEY = "history";
	protected static final String CARROT_HISTORY_NBT_KEY = "carrotHistory";

	public static FoodHistory get(class_1657 player) {
		if (player == null) {
			return null;
		}
		class_1702 hungerManager = player.method_7344();
		if (!(hungerManager instanceof IHungerManager)) {
			return null;
		}
		return ((IHungerManager) hungerManager).spiceOfFabric_getFoodHistory();
	}

	protected final BiMap<Integer, FoodHistoryEntry> dictionary;
	protected int nextId = 0;

	protected final FixedLengthIntFIFOQueueWithStats recentlyEaten;

	protected final Set<FoodHistoryEntry> uniqueFoodsEaten;

	public FoodHistory() {
		dictionary = HashBiMap.create();
		recentlyEaten = new FixedLengthIntFIFOQueueWithStats(new ArrayFixedLengthIntFIFOQueue(SOFConfig.food.historyLength));
		uniqueFoodsEaten = new HashSet<>();
	}

	public void reset() {
		resetHistory();
		resetUniqueFoodsEaten();
	}

	public void resetHistory() {
		dictionary.clear();
		nextId = 0;
		recentlyEaten.clear();
	}

	public Set<FoodHistoryEntry> getUniqueFoodsEaten() {
		return uniqueFoodsEaten;
	}

	public void resetUniqueFoodsEaten() {
		uniqueFoodsEaten.clear();
	}

	public void write(class_2540 buffer) {
		buffer.method_10804(dictionary.size());
		for (Map.Entry<Integer, FoodHistoryEntry> entry : dictionary.entrySet()) {
			buffer.method_10804(entry.getKey());
			entry.getValue().write(buffer);
		}
		buffer.method_10804(recentlyEaten.size());
		for (int integer : recentlyEaten) {
			buffer.method_10804(integer);
		}
		if (SOFConfig.carrot.enable) {
			buffer.method_52964(true);
			buffer.method_10804(uniqueFoodsEaten.size());
			for (FoodHistoryEntry entry : uniqueFoodsEaten) {
				entry.write(buffer);
			}
		} else {
			buffer.method_52964(false);
		}
	}

	public void read(class_2540 buffer) {
		dictionary.clear();
		recentlyEaten.clear();
		recentlyEaten.setLength(SOFConfig.food.historyLength);

		for (int l = buffer.method_10816(), i = 0; i < l; i++) {
			dictionary.put(buffer.method_10816(), FoodHistoryEntry.from(buffer));
		}
		for (int l = buffer.method_10816(), i = 0; i < l; i++) {
			// Using forceEnqueue here to make sure we're not running out of space and throwing an exception
			// just because of a small desync of the history length ;)
			recentlyEaten.forceEnqueue(buffer.method_10816());
		}

		uniqueFoodsEaten.clear();

		if (buffer.readBoolean()) {
			final int length = buffer.method_10816();
			for (int i = 0; i < length; i++) {
				uniqueFoodsEaten.add(FoodHistoryEntry.from(buffer));
			}
		}
	}

	public class_2487 write(class_2487 compoundTag) {
		defragmentDictionary();
		class_2499 list = new class_2499();
		for (Map.Entry<Integer, FoodHistoryEntry> entry : dictionary.entrySet()) {
			list.method_10531(entry.getKey(), entry.getValue().write(new class_2487()));
		}
		compoundTag.method_10566(DICTIONARY_NBT_KEY, list);
		class_2499 historyList = new class_2499();
		for (Integer id : recentlyEaten) {
			historyList.add(class_2497.method_23247(id));
		}
		compoundTag.method_10566(RECENT_HISTORY_NBT_KEY, historyList);
		class_2499 carrotHistoryList = new class_2499();
		for (FoodHistoryEntry entry : uniqueFoodsEaten) {
			carrotHistoryList.add(entry.write(new class_2487()));
		}
		compoundTag.method_10566(CARROT_HISTORY_NBT_KEY, carrotHistoryList);
		return compoundTag;
	}

	public static FoodHistory read(class_2487 compoundTag) {
		FoodHistory foodHistory = new FoodHistory();
		if (compoundTag.method_10573(DICTIONARY_NBT_KEY, 9)) {
			class_2499 nbtDictionary = compoundTag.method_10554(DICTIONARY_NBT_KEY, 10);
			for (int i = 0; i < nbtDictionary.size(); i++) {
				FoodHistoryEntry entry = new FoodHistoryEntry().read((class_2487) nbtDictionary.method_10534(i));
				if (entry != null) {
					foodHistory.dictionary.put(i, entry);
				}
			}
		}
		foodHistory.nextId = foodHistory.dictionary.size();

		if (compoundTag.method_10573(RECENT_HISTORY_NBT_KEY, 9)) {
			class_2499 nbtRecentHistory = compoundTag.method_10554(RECENT_HISTORY_NBT_KEY, 3);

			for (class_2520 tag : nbtRecentHistory) {
				// Using forceEnqueue here to make sure we're not running out of space and throwing an exception.
				// The history length might have changed (decreased) since the last time the player logged in.
				foodHistory.recentlyEaten.forceEnqueue(((class_2497) tag).method_10701());
			}
		}

		if (compoundTag.method_10573(CARROT_HISTORY_NBT_KEY, 9)) {
			class_2499 nbtCarrotHistory = compoundTag.method_10554(CARROT_HISTORY_NBT_KEY, 10);
			for (class_2520 tag : nbtCarrotHistory) {
				if (!(tag instanceof class_2487 carrotEntry)) {
					continue;
				}
				FoodHistoryEntry entry = new FoodHistoryEntry().read(carrotEntry);
				if (entry != null) {
					foodHistory.uniqueFoodsEaten.add(entry);
				}
			}
		}

		return foodHistory;
	}

	public void defragmentDictionary() {
		Int2IntMap oldToNewMap = new Int2IntArrayMap();
		int i = 0;
		for (Integer id : dictionary.keySet()) {
			oldToNewMap.put((int) id, i);
			i++;
		}
		nextId = i;
		int historySize = recentlyEaten.size();
		for (int j = 0; j < historySize; j++) {
			recentlyEaten.enqueue(oldToNewMap.get(recentlyEaten.dequeue()));
		}
		BiMap<Integer, FoodHistoryEntry> newDictionary = HashBiMap.create();
		for (Map.Entry<Integer, FoodHistoryEntry> entry : dictionary.entrySet()) {
			newDictionary.put(oldToNewMap.get((int) entry.getKey()), entry.getValue());
		}
		dictionary.clear();
		dictionary.putAll(newDictionary);
	}

	public int getTimesRecentlyEaten(class_1799 stack) {
		Integer id = dictionary.inverse().get(FoodHistoryEntry.fromItemStack(stack));
		if (id == null) {
			return 0;
		}
		return recentlyEaten.getStats().getOrDefault((int) id, 0);
	}

	public int getFoodCountSinceLastEaten(class_1799 stack) {
		Integer id = dictionary.inverse().get(FoodHistoryEntry.fromItemStack(stack));
		if (id == null) {
			return -1;
		}
		IntIterator iterator = recentlyEaten.iterator();
		int foundI = Integer.MIN_VALUE;
		while (iterator.hasNext()) {
			foundI++;
			if (iterator.nextInt() == id) {
				foundI = 0;
			}
		}
		return foundI;
	}

	public void addFood(class_1799 stack, class_3222 player) {
		FoodHistoryEntry entry = FoodHistoryEntry.fromItemStack(stack);

		if (SpiceOfFabric.hasClientMod(player)) {
			SOFCommonNetworking.sendAddFoodPacket(player, entry);
		}

		addFood(entry);
	}

	public void addFood(FoodHistoryEntry entry) {
		Integer boxedId = dictionary.inverse().get(entry);
		int id;
		if (boxedId == null) {
			id = nextId++;
			dictionary.put(id, entry);
		} else {
			id = boxedId;
		}

		// Make sure the history length is correct, just in case
		if (recentlyEaten.getLength() != SOFConfig.food.historyLength) {
			recentlyEaten.setLength(SOFConfig.food.historyLength);
		}

		recentlyEaten.forceEnqueue(id);

		if (SOFConfig.carrot.enable) {
			uniqueFoodsEaten.add(entry);
		}
	}

	public int getRecentlyEatenCount() {
		return recentlyEaten.size();
	}

	public class_1799 getStackFromRecentlyEaten(int index) {
		if (index < 0 || index >= recentlyEaten.size()) {
			return null;
		}
		return dictionary.get(recentlyEaten.get(index)).getStack();
	}

	public boolean isInUniqueEaten(class_1799 stack) {
		FoodHistoryEntry entry = FoodHistoryEntry.fromItemStack(stack);
		return uniqueFoodsEaten.contains(entry);
	}

	public int getCarrotHealthOffset(class_1657 player) {
		class_1324 maxHealthAttr = player.method_5996(class_5134.field_23716);
		SOFConfig.setHealthFormulaExpressionValues(uniqueFoodsEaten.size(), (int) maxHealthAttr.method_6201());

		int newMaxHealth = class_3532.method_15357(SOFConfig.healthFormulaExpression.evaluate());
		if (SOFConfig.carrot.maxHealth > 0) {
			newMaxHealth = class_3532.method_15340(newMaxHealth, 1, SOFConfig.carrot.maxHealth);
		}
		return newMaxHealth - (int) maxHealthAttr.method_6201();
	}

	public int getCarrotMaxHealth(class_1657 player) {
		class_1324 maxHealthAttr = player.method_5996(class_5134.field_23716);
		SOFConfig.setHealthFormulaExpressionValues(uniqueFoodsEaten.size(), (int) maxHealthAttr.method_6201());
		return class_3532.method_15357(SOFConfig.healthFormulaExpression.evaluate());
	}
}
