package de.siphalor.tweed.tailor;

import com.terraformersmc.modmenu.api.ConfigScreenFactory;
import de.siphalor.tweed.Tweed;
import de.siphalor.tweed.client.CustomNoticeScreen;
import de.siphalor.tweed.client.TweedClient;
import de.siphalor.tweed.client.cloth.ClothDropdownSelectEntry;
import de.siphalor.tweed.config.*;
import de.siphalor.tweed.config.constraints.ConstraintException;
import de.siphalor.tweed.config.entry.ConfigEntry;
import de.siphalor.tweed.config.entry.ValueConfigEntry;
import io.netty.buffer.Unpooled;
import me.shedaniel.clothconfig2.api.AbstractConfigListEntry;
import me.shedaniel.clothconfig2.api.ConfigBuilder;
import me.shedaniel.clothconfig2.api.ConfigEntryBuilder;
import me.shedaniel.clothconfig2.impl.builders.*;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.fabricmc.fabric.api.client.networking.v1.ClientPlayNetworking;
import net.minecraft.class_1074;
import net.minecraft.class_2540;
import net.minecraft.class_2588;
import net.minecraft.class_310;
import net.minecraft.class_437;
import java.awt.*;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;

@Environment(EnvType.CLIENT)
public class ClothTailor extends Tailor {
	public static final ClothTailor INSTANCE = new ClothTailor();

	protected static final String SCREEN_NAME_PREFIX = "tweed.cloth.";
	protected boolean waitingForFile;

	private static final Map<Class<?>, EntryConverter<?>> ENTRY_CONVERTERS = new HashMap<>();
	private final Map<String, ConfigScreenFactory<?>> screenFactories = new HashMap<>();

	@Override
	public void process(ConfigFile configFile) {
		String modId = configFile.getName();
		ClothData clothData = configFile.getTailorAnnotation(ClothData.class);
		if (clothData != null && !clothData.modid().isEmpty()) {
			modId = clothData.modid();
		}
		screenFactories.put(modId, parent -> convert(configFile, parent));
	}

	public Map<String, ConfigScreenFactory<?>> getScreenFactories() {
		return screenFactories;
	}

	public class_437 convert(ConfigFile configFile, class_437 parentScreen) {
		boolean inGame = class_310.method_1551().field_1687 != null;
		if (inGame && configFile.getRootCategory().getEnvironment() != ConfigEnvironment.CLIENT) {
			return new CustomNoticeScreen(
					() -> {
						waitingForFile = true;

						class_2540 buf = new class_2540(Unpooled.buffer());
						buf.method_10814(configFile.getName());
						buf.method_10817(ConfigEnvironment.UNIVERSAL);
						buf.method_10817(ConfigScope.SMALLEST);
						buf.method_10817(ConfigOrigin.MAIN);
						ClientPlayNetworking.send(Tweed.REQUEST_SYNC_C2S_PACKET, buf);

						TweedClient.setSyncRunnable(() -> {
							if (waitingForFile) {
								waitingForFile = false;
								class_310.method_1551().method_1507(buildConfigScreen(configFile, parentScreen));
							}
						});
					},
					() -> {
						waitingForFile = false;
						class_310.method_1551().method_1507(parentScreen);
					},
					new class_2588("tweed.gui.screen.syncFromServer"),
					new class_2588("tweed.gui.screen.syncFromServer.note")
			);
		} else {
			return buildConfigScreen(configFile, parentScreen);
		}
	}

	public class_437 buildConfigScreen(ConfigFile configFile, class_437 parentScreen) {
		final String path = SCREEN_NAME_PREFIX + configFile.getName();
		ConfigBuilder configBuilder = ConfigBuilder.create();
		configBuilder.setParentScreen(parentScreen);
		if (configFile.getRootCategory().getBackgroundTexture() != null) {
			configBuilder.setDefaultBackgroundTexture(configFile.getRootCategory().getBackgroundTexture());
		}
		configBuilder.setSavingRunnable(() -> save(configFile));
		configBuilder.setTitle(class_1074.method_4662(path));

		if (configFile.getRootCategory().entryStream().allMatch(entry -> entry.getValue() instanceof ConfigCategory)) {
			configFile.getRootCategory().entryStream().forEach(entry -> createCategory(configBuilder, (ConfigCategory) entry.getValue(), path + "." + entry.getKey()));
		} else {
			createCategory(configBuilder, configFile.getRootCategory(), path + ".main");
		}

		return configBuilder.build();
	}

	private void convertCategory(ConfigEntryBuilder entryBuilder, Consumer<AbstractConfigListEntry<?>> registry, ConfigCategory configCategory, String path) {
		configCategory.entryStream().forEachOrdered(entry -> {
			final String subPath = path + "." + entry.getKey();

			if (entry.getValue() instanceof ConfigCategory) {
				SubCategoryBuilder categoryBuilder = entryBuilder.startSubCategory(class_1074.method_4662(subPath));
				categoryBuilder.add(entryBuilder.startTextDescription(categoryDescription(subPath, entry.getValue())).setColor(Color.GRAY.getRGB()).build());

				convertCategory(entryBuilder, categoryBuilder::add, (ConfigCategory) entry.getValue(), subPath);

				registry.accept(categoryBuilder.build());

			} else if (entry.getValue() instanceof ValueConfigEntry<?>) {
				Class<?> clazz = ((ValueConfigEntry<?>) entry.getValue()).getType();
				EntryConverter<?> entryConverter;

				entryConverter = ENTRY_CONVERTERS.get(clazz);
				main:
				while (clazz != Object.class && entryConverter == null) {
					for (Class<?> anInterface : clazz.getInterfaces()) {
						entryConverter = ENTRY_CONVERTERS.get(anInterface);
						if (entryConverter != null) {
							break main;
						}
					}

					clazz = clazz.getSuperclass();
					entryConverter = ENTRY_CONVERTERS.get(clazz);
				}

				if (entryConverter != null) {
					//noinspection unchecked,rawtypes
					registry.accept(entryConverter.convert((ValueConfigEntry) entry.getValue(), entryBuilder, subPath));
				} else {
					Tweed.LOGGER.warn("Couldn't convert config entry of type " + ((ValueConfigEntry<?>) entry.getValue()).getType().getSimpleName() + " to cloth entry!");
				}
			}
		});
	}

	private void createCategory(ConfigBuilder configBuilder, ConfigCategory configCategory, String name) {
		me.shedaniel.clothconfig2.api.ConfigCategory clothCategory = configBuilder.getOrCreateCategory(class_1074.method_4662(name));
		if (configCategory.getBackgroundTexture() != null) {
			clothCategory.setCategoryBackground(configCategory.getBackgroundTexture());
		}
		clothCategory.addEntry(configBuilder.entryBuilder().startTextDescription(categoryDescription(name, configCategory)).setColor(Color.GRAY.getRGB()).build());
		convertCategory(configBuilder.entryBuilder(), clothCategory::addEntry, configCategory, name);
	}

	private void save(ConfigFile configFile) {
		if (TweedClient.isOnRemoteServer()) {
			configFile.syncToServer(ConfigEnvironment.UNIVERSAL, ConfigScope.SMALLEST);
			ConfigLoader.updateMainConfigFile(configFile, ConfigEnvironment.UNIVERSAL, ConfigScope.HIGHEST);
			ConfigLoader.loadConfigs(class_310.method_1551().method_1478(), ConfigEnvironment.UNIVERSAL, ConfigScope.SMALLEST);
		} else {
			ConfigLoader.updateMainConfigFile(configFile, ConfigEnvironment.UNIVERSAL, ConfigScope.HIGHEST);
			ConfigLoader.loadConfigs(class_310.method_1551().method_1478(), ConfigEnvironment.UNIVERSAL, ConfigScope.WORLD);
		}
	}

	public static <V> void registerEntryConverter(Class<V> valueType, EntryConverter<V> converter) {
		ENTRY_CONVERTERS.put(valueType, converter);
	}

	public static <V> Optional<String> errorSupplier(V value, ValueConfigEntry<V> configEntry) {
		try {
			configEntry.applyConstraints(value);
		} catch (ConstraintException e) {
			return Optional.of(e.getMessage());
		}
		return Optional.empty();
	}

	public static boolean requiresRestart(ValueConfigEntry<?> configEntry) {
		if (class_310.method_1551().field_1687 == null) {
			return configEntry.getScope().triggers(ConfigScope.GAME);
		} else {
			return configEntry.getScope().triggers(ConfigScope.WORLD);
		}
	}

	public static Optional<String[]> description(String langKey, ValueConfigEntry<?> configEntry) {
		if (class_1074.method_4663(langKey + ".description")) {
			return Optional.of(new String[]{class_1074.method_4662(langKey + ".description")});
		}
		return configEntry.getClothyDescription();
	}

	public static String categoryDescription(String langKey, ConfigEntry<?> entry) {
		if (class_1074.method_4663(langKey + ".description")) {
			return class_1074.method_4662(langKey + ".description");
		}
		return entry.getDescription().replace("\t", "    ");
	}

	@FunctionalInterface
	public interface EntryConverter<V> {
		AbstractConfigListEntry<?> convert(ValueConfigEntry<V> configEntry, ConfigEntryBuilder entryBuilder, String langKey);
	}

	static {
		registerEntryConverter(Boolean.class, (configEntry, entryBuilder, langKey) -> {
					BooleanToggleBuilder builder = entryBuilder.startBooleanToggle(class_1074.method_4662(langKey), configEntry.getMainConfigValue());
					builder.setDefaultValue(configEntry::getDefaultValue);
					builder.setSaveConsumer(configEntry::setMainConfigValue);
					builder.setTooltip(description(langKey, configEntry));
					builder.setErrorSupplier(value -> errorSupplier(value, configEntry));
					if (requiresRestart(configEntry)) {
						builder.requireRestart(true);
					}
					return builder.build();
				}
		);
		//noinspection unchecked,rawtypes
		registerEntryConverter(DropdownMaterial.class, (configEntry, entryBuilder, langKey) ->
				new ClothDropdownSelectEntry<>(
						class_1074.method_4662(langKey),
						configEntry.getMainConfigValue(),
						class_1074.method_4662("text.cloth-config.reset_value"),
						() -> description(langKey, configEntry),
						requiresRestart(configEntry),
						configEntry::getDefaultValue,
						configEntry::setMainConfigValue,
						new ArrayList<DropdownMaterial>(configEntry.getDefaultValue().values()),
						dropdownMaterial -> new class_2588(dropdownMaterial.getTranslationKey())
				)
		);
		registerEntryConverter(Enum.class, (configEntry, entryBuilder, langKey) -> {
					//noinspection unchecked
					EnumSelectorBuilder<Enum<?>> builder = (EnumSelectorBuilder<Enum<?>>) (Object) entryBuilder.startEnumSelector(class_1074.method_4662(langKey), configEntry.getType(), configEntry.getMainConfigValue());
					builder.setDefaultValue(configEntry::getDefaultValue);
					builder.setSaveConsumer(configEntry::setMainConfigValue);
					builder.setTooltip(description(langKey, configEntry));
					builder.setErrorSupplier(value -> errorSupplier(value, configEntry));
					if (requiresRestart(configEntry)) {
						builder.requireRestart(true);
					}
					return builder.build();
				}
		);
		registerEntryConverter(Float.class, (configEntry, entryBuilder, langKey) -> {
					FloatFieldBuilder builder = entryBuilder.startFloatField(class_1074.method_4662(langKey), configEntry.getMainConfigValue());
					builder.setDefaultValue(configEntry::getDefaultValue);
					builder.setSaveConsumer(configEntry::setMainConfigValue);
					builder.setTooltip(description(langKey, configEntry));
					builder.setErrorSupplier(value -> errorSupplier(value, configEntry));
					if (requiresRestart(configEntry)) {
						builder.requireRestart(true);
					}
					return builder.build();
				}
		);
		registerEntryConverter(Integer.class, (configEntry, entryBuilder, langKey) -> {
					IntFieldBuilder builder = entryBuilder.startIntField(class_1074.method_4662(langKey), configEntry.getMainConfigValue());
					builder.setDefaultValue(configEntry::getDefaultValue);
					builder.setSaveConsumer(configEntry::setMainConfigValue);
					builder.setTooltip(description(langKey, configEntry));
					builder.setErrorSupplier(value -> errorSupplier(value, configEntry));
					if (requiresRestart(configEntry)) {
						builder.requireRestart(true);
					}
					return builder.build();
				}
		);
		registerEntryConverter(String.class, (configEntry, entryBuilder, langKey) -> {
					StringFieldBuilder builder = entryBuilder.startStrField(class_1074.method_4662(langKey), configEntry.getMainConfigValue());
					builder.setDefaultValue(configEntry::getDefaultValue);
					builder.setSaveConsumer(configEntry::setMainConfigValue);
					builder.setTooltip(description(langKey, configEntry));
					builder.setErrorSupplier(value -> errorSupplier(value, configEntry));
					if (requiresRestart(configEntry)) {
						builder.requireRestart(true);
					}
					return builder.build();
				}
		);
	}
}
