package de.siphalor.tweed5.data.extension.impl;

import com.google.auto.service.AutoService;
import de.siphalor.tweed5.core.api.container.ConfigContainer;
import de.siphalor.tweed5.core.api.entry.ConfigEntry;
import de.siphalor.tweed5.core.api.extension.TweedExtension;
import de.siphalor.tweed5.core.api.extension.TweedExtensionSetupContext;
import de.siphalor.tweed5.core.api.middleware.DefaultMiddlewareContainer;
import de.siphalor.tweed5.core.api.middleware.Middleware;
import de.siphalor.tweed5.data.extension.api.*;
import de.siphalor.tweed5.data.extension.api.extension.ReadWriteExtensionSetupContext;
import de.siphalor.tweed5.data.extension.api.extension.ReadWriteRelatedExtension;
import de.siphalor.tweed5.dataapi.api.TweedDataReader;
import de.siphalor.tweed5.dataapi.api.TweedDataVisitor;
import de.siphalor.tweed5.dataapi.api.TweedDataWriteException;
import de.siphalor.tweed5.patchwork.api.Patchwork;
import de.siphalor.tweed5.patchwork.api.PatchworkFactory;
import de.siphalor.tweed5.patchwork.api.PatchworkPartAccess;
import org.jspecify.annotations.Nullable;
import java.util.Collection;

@AutoService(ReadWriteExtension.class)
public class ReadWriteExtensionImpl implements ReadWriteExtension {
	private final ConfigContainer<?> configContainer;
	private final PatchworkPartAccess<CustomEntryData> customEntryDataAccess;
	private DefaultMiddlewareContainer<TweedEntryReader<?, ?>> entryReaderMiddlewareContainer = new DefaultMiddlewareContainer<>();
	private DefaultMiddlewareContainer<TweedEntryWriter<?, ?>> entryWriterMiddlewareContainer = new DefaultMiddlewareContainer<>();
	@Nullable
	private PatchworkFactory readWriteContextPatchworkFactory;

	public ReadWriteExtensionImpl(ConfigContainer<?> configContainer, TweedExtensionSetupContext context) {
		this.configContainer = configContainer;
		this.customEntryDataAccess = context.registerEntryExtensionData(CustomEntryData.class);
	}

	@Override
	public void extensionsFinalized() {
		Collection<TweedExtension> extensions = configContainer.extensions();
		PatchworkFactory.Builder readWriteContextPatchworkFactorBuilder = PatchworkFactory.builder();
		entryReaderMiddlewareContainer = new DefaultMiddlewareContainer<>();
		entryWriterMiddlewareContainer = new DefaultMiddlewareContainer<>();
		ReadWriteExtensionSetupContext setupContext = new ReadWriteExtensionSetupContext() {
			@Override
			public <E> PatchworkPartAccess<E> registerReadWriteContextExtensionData(Class<E> extensionDataClass) {
				return readWriteContextPatchworkFactorBuilder.registerPart(extensionDataClass);
			}
			@Override
			public void registerReaderMiddleware(Middleware<TweedEntryReader<?, ?>> middleware) {
				entryReaderMiddlewareContainer.register(middleware);
			}
			@Override
			public void registerWriterMiddleware(Middleware<TweedEntryWriter<?, ?>> middleware) {
				entryWriterMiddlewareContainer.register(middleware);
			}
		};
		for (TweedExtension extension : extensions) {
			if (extension instanceof ReadWriteRelatedExtension) {
				((ReadWriteRelatedExtension) extension).setupReadWriteExtension(setupContext);
			}
		}
		readWriteContextPatchworkFactory = readWriteContextPatchworkFactorBuilder.build();
		entryReaderMiddlewareContainer.seal();
		entryWriterMiddlewareContainer.seal();
	}

	@Override
	@Nullable
	public <T, C extends ConfigEntry<T>> TweedEntryReader<T, C> getDefinedEntryReader(ConfigEntry<T> entry) {
		CustomEntryData customEntryData = entry.extensionsData().get(customEntryDataAccess);
		if (customEntryData == null) {
			return null;
		}
		//noinspection unchecked
		return (TweedEntryReader<T, C>) customEntryData.readerDefinition();
	}

	@Override
	@Nullable
	public <T, C extends ConfigEntry<T>> TweedEntryWriter<T, C> getDefinedEntryWriter(ConfigEntry<T> entry) {
		CustomEntryData customEntryData = entry.extensionsData().get(customEntryDataAccess);
		if (customEntryData == null) {
			return null;
		}
		//noinspection unchecked
		return (TweedEntryWriter<T, C>) customEntryData.writerDefinition();
	}

	@Override
	public <T, C extends ConfigEntry<T>> void setEntryReaderWriter(ConfigEntry<T> entry, TweedEntryReader<T, C> entryReader, TweedEntryWriter<T, C> entryWriter) {
		CustomEntryData customEntryData = getOrCreateCustomEntryData(entry);
		customEntryData.readerDefinition(entryReader);
		customEntryData.writerDefinition(entryWriter);
	}

	@Override
	public <T, C extends ConfigEntry<T>> void setEntryReader(ConfigEntry<T> entry, TweedEntryReader<T, C> entryReader) {
		getOrCreateCustomEntryData(entry).readerDefinition(entryReader);
	}

	@Override
	public <T, C extends ConfigEntry<T>> void setEntryWriter(ConfigEntry<T> entry, TweedEntryWriter<T, C> entryWriter) {
		getOrCreateCustomEntryData(entry).writerDefinition(entryWriter);
	}

	@Override
	public void initEntry(ConfigEntry<?> configEntry) {
		CustomEntryData customEntryData = getOrCreateCustomEntryData(configEntry);
		customEntryData.readerChain(entryReaderMiddlewareContainer.process(customEntryData.readerDefinition()));
		customEntryData.writerChain(entryWriterMiddlewareContainer.process(customEntryData.writerDefinition()));
	}

	private CustomEntryData getOrCreateCustomEntryData(ConfigEntry<?> entry) {
		CustomEntryData entryData = entry.extensionsData().get(customEntryDataAccess);
		if (entryData == null) {
			entryData = new CustomEntryData();
			entry.extensionsData().set(customEntryDataAccess, entryData);
		}
		return entryData;
	}

	@Override
	public Patchwork createReadWriteContextExtensionsData() {
		assert readWriteContextPatchworkFactory != null;
		return readWriteContextPatchworkFactory.create();
	}

	public <T extends @Nullable Object> T read(TweedDataReader reader, ConfigEntry<T> entry, Patchwork contextExtensionsData) throws TweedEntryReadException {
		TweedReadContext context = new TweedReadWriteContextImpl(this, contextExtensionsData);
		return getReaderChain(entry).read(reader, entry, context);
	}

	@Override
	public <T extends @Nullable Object> void write(TweedDataVisitor writer, @Nullable T value, ConfigEntry<T> entry, Patchwork contextExtensionsData) throws TweedEntryWriteException {
		TweedWriteContext context = new TweedReadWriteContextImpl(this, contextExtensionsData);
		try {
			getWriterChain(entry).write(writer, value, entry, context);
		} catch (TweedDataWriteException e) {
			throw new TweedEntryWriteException("Failed to write entry", e, context);
		}
	}


	private static class CustomEntryData {
		private TweedEntryReader<?, ?> readerDefinition = TweedEntryReaderWriterImpls.NOOP_READER_WRITER;
		private TweedEntryWriter<?, ?> writerDefinition = TweedEntryReaderWriterImpls.NOOP_READER_WRITER;
		private TweedEntryReader<?, ?> readerChain;
		private TweedEntryWriter<?, ?> writerChain;

		public CustomEntryData() {
		}

		public TweedEntryReader<?, ?> readerDefinition() {
			return this.readerDefinition;
		}

		public TweedEntryWriter<?, ?> writerDefinition() {
			return this.writerDefinition;
		}

		public TweedEntryReader<?, ?> readerChain() {
			return this.readerChain;
		}

		public TweedEntryWriter<?, ?> writerChain() {
			return this.writerChain;
		}

		/**
		 * @return {@code this}.
		 */
		public ReadWriteExtensionImpl.CustomEntryData readerDefinition(final TweedEntryReader<?, ?> readerDefinition) {
			this.readerDefinition = readerDefinition;
			return this;
		}

		/**
		 * @return {@code this}.
		 */
		public ReadWriteExtensionImpl.CustomEntryData writerDefinition(final TweedEntryWriter<?, ?> writerDefinition) {
			this.writerDefinition = writerDefinition;
			return this;
		}

		/**
		 * @return {@code this}.
		 */
		public ReadWriteExtensionImpl.CustomEntryData readerChain(final TweedEntryReader<?, ?> readerChain) {
			this.readerChain = readerChain;
			return this;
		}

		/**
		 * @return {@code this}.
		 */
		public ReadWriteExtensionImpl.CustomEntryData writerChain(final TweedEntryWriter<?, ?> writerChain) {
			this.writerChain = writerChain;
			return this;
		}

		@Override
		public boolean equals(final Object o) {
			if (o == this) return true;
			if (!(o instanceof ReadWriteExtensionImpl.CustomEntryData)) return false;
			final ReadWriteExtensionImpl.CustomEntryData other = (ReadWriteExtensionImpl.CustomEntryData) o;
			if (!other.canEqual((Object) this)) return false;
			final Object this$readerDefinition = this.readerDefinition();
			final Object other$readerDefinition = other.readerDefinition();
			if (this$readerDefinition == null ? other$readerDefinition != null : !this$readerDefinition.equals(other$readerDefinition)) return false;
			final Object this$writerDefinition = this.writerDefinition();
			final Object other$writerDefinition = other.writerDefinition();
			if (this$writerDefinition == null ? other$writerDefinition != null : !this$writerDefinition.equals(other$writerDefinition)) return false;
			final Object this$readerChain = this.readerChain();
			final Object other$readerChain = other.readerChain();
			if (this$readerChain == null ? other$readerChain != null : !this$readerChain.equals(other$readerChain)) return false;
			final Object this$writerChain = this.writerChain();
			final Object other$writerChain = other.writerChain();
			if (this$writerChain == null ? other$writerChain != null : !this$writerChain.equals(other$writerChain)) return false;
			return true;
		}

		protected boolean canEqual(final Object other) {
			return other instanceof ReadWriteExtensionImpl.CustomEntryData;
		}

		@Override
		public int hashCode() {
			final int PRIME = 59;
			int result = 1;
			final Object $readerDefinition = this.readerDefinition();
			result = result * PRIME + ($readerDefinition == null ? 43 : $readerDefinition.hashCode());
			final Object $writerDefinition = this.writerDefinition();
			result = result * PRIME + ($writerDefinition == null ? 43 : $writerDefinition.hashCode());
			final Object $readerChain = this.readerChain();
			result = result * PRIME + ($readerChain == null ? 43 : $readerChain.hashCode());
			final Object $writerChain = this.writerChain();
			result = result * PRIME + ($writerChain == null ? 43 : $writerChain.hashCode());
			return result;
		}

		@Override
		public String toString() {
			return "ReadWriteExtensionImpl.CustomEntryData(readerDefinition=" + this.readerDefinition() + ", writerDefinition=" + this.writerDefinition() + ", readerChain=" + this.readerChain() + ", writerChain=" + this.writerChain() + ")";
		}
	}

	@Override
	public <T, C extends ConfigEntry<T>> TweedEntryReader<T, C> getReaderChain(C entry) {
		//noinspection unchecked
		return (TweedEntryReader<T, C>) entry.extensionsData().get(customEntryDataAccess).readerChain();
	}

	@Override
	public <T, C extends ConfigEntry<T>> TweedEntryWriter<T, C> getWriterChain(C entry) {
		//noinspection unchecked
		return (TweedEntryWriter<T, C>) entry.extensionsData().get(customEntryDataAccess).writerChain();
	}
}
