package de.siphalor.tweed5.defaultextensions.comment.impl;

import de.siphalor.tweed5.core.api.entry.ConfigEntry;
import de.siphalor.tweed5.core.api.middleware.Middleware;
import de.siphalor.tweed5.data.extension.api.TweedEntryWriter;
import de.siphalor.tweed5.dataapi.api.DelegatingTweedDataWriter;
import de.siphalor.tweed5.dataapi.api.TweedDataVisitor;
import de.siphalor.tweed5.dataapi.api.TweedDataWriter;
import de.siphalor.tweed5.dataapi.api.decoration.TweedDataCommentDecoration;
import de.siphalor.tweed5.dataapi.api.decoration.TweedDataDecoration;
import de.siphalor.tweed5.patchwork.api.PatchworkPartAccess;
import org.jspecify.annotations.NonNull;
import org.jspecify.annotations.Nullable;
import java.util.ArrayDeque;
import java.util.Deque;

class TweedEntryWriterCommentMiddleware implements Middleware<TweedEntryWriter<?, ?>> {
	private final CommentExtensionImpl commentExtension;

	@Override
	public String id() {
		return "comment-writer";
	}

	@Override
	public TweedEntryWriter<?, ?> process(TweedEntryWriter<?, ?> inner) {
		PatchworkPartAccess<Boolean> writerInstalledAccess = commentExtension.writerInstalledReadWriteContextAccess();
		assert writerInstalledAccess != null;
		//noinspection unchecked
		TweedEntryWriter<Object, ConfigEntry<Object>> innerCasted = (TweedEntryWriter<Object, @NonNull ConfigEntry<Object>>) inner;
		return (TweedEntryWriter<Object, @NonNull ConfigEntry<Object>>) (writer, value, entry, context) -> {
			if (!Boolean.TRUE.equals(context.extensionsData().get(writerInstalledAccess))) {
				context.extensionsData().set(writerInstalledAccess, Boolean.TRUE);
				writer = new MapEntryKeyDeferringWriter(writer);
			}
			String comment = commentExtension.getFullComment(entry);
			if (comment != null) {
				writer.visitDecoration(new PiercingCommentDecoration(() -> comment));
			}
			innerCasted.write(writer, value, entry, context);
		};
	}


	private static class MapEntryKeyDeferringWriter extends DelegatingTweedDataWriter {
		private final Deque<TweedDataDecoration> decorationQueue = new ArrayDeque<>();
		@Nullable
		private String mapEntryKey;

		protected MapEntryKeyDeferringWriter(TweedDataVisitor delegate) {
			super(delegate);
		}

		@Override
		public void visitMapEntryKey(String key) {
			if (mapEntryKey != null) {
				throw new IllegalStateException("Map entry key already visited");
			} else {
				mapEntryKey = key;
			}
		}

		@Override
		public void visitDecoration(TweedDataDecoration decoration) {
			if (decoration instanceof PiercingCommentDecoration) {
				super.visitDecoration(((PiercingCommentDecoration) decoration).commentDecoration());
				return;
			}
			if (mapEntryKey != null) {
				decorationQueue.addLast(decoration);
			} else {
				super.visitDecoration(decoration);
			}
		}

		@Override
		protected void beforeValueWrite() {
			if (mapEntryKey != null) {
				super.visitMapEntryKey(mapEntryKey);
				mapEntryKey = null;
				TweedDataDecoration decoration;
				while ((decoration = decorationQueue.pollFirst()) != null) {
					super.visitDecoration(decoration);
				}
			}
			super.beforeValueWrite();
		}
	}


	private static final class PiercingCommentDecoration implements TweedDataDecoration {
		private final TweedDataCommentDecoration commentDecoration;

		public PiercingCommentDecoration(final TweedDataCommentDecoration commentDecoration) {
			this.commentDecoration = commentDecoration;
		}

		public TweedDataCommentDecoration commentDecoration() {
			return this.commentDecoration;
		}

		@Override
		public boolean equals(final Object o) {
			if (o == this) return true;
			if (!(o instanceof TweedEntryWriterCommentMiddleware.PiercingCommentDecoration)) return false;
			final TweedEntryWriterCommentMiddleware.PiercingCommentDecoration other = (TweedEntryWriterCommentMiddleware.PiercingCommentDecoration) o;
			final Object this$commentDecoration = this.commentDecoration();
			final Object other$commentDecoration = other.commentDecoration();
			if (this$commentDecoration == null ? other$commentDecoration != null : !this$commentDecoration.equals(other$commentDecoration)) return false;
			return true;
		}

		@Override
		public int hashCode() {
			final int PRIME = 59;
			int result = 1;
			final Object $commentDecoration = this.commentDecoration();
			result = result * PRIME + ($commentDecoration == null ? 43 : $commentDecoration.hashCode());
			return result;
		}

		@Override
		public String toString() {
			return "TweedEntryWriterCommentMiddleware.PiercingCommentDecoration(commentDecoration=" + this.commentDecoration() + ")";
		}
	}

	public TweedEntryWriterCommentMiddleware(final CommentExtensionImpl commentExtension) {
		this.commentExtension = commentExtension;
	}
}
