package de.siphalor.tweed5.coat.bridge.impl;

import de.siphalor.coat.handler.ConfigEntryHandler;
import de.siphalor.coat.input.CheckBoxConfigInput;
import de.siphalor.coat.input.ConfigInput;
import de.siphalor.coat.input.CycleButtonConfigInput;
import de.siphalor.coat.input.TextConfigInput;
import de.siphalor.coat.list.complex.ConfigCategoryWidget;
import de.siphalor.coat.list.entry.ConfigCategoryConfigEntry;
import de.siphalor.coat.screen.ConfigContentWidget;
import de.siphalor.coat.util.EnumeratedMaterial;
import de.siphalor.tweed5.attributesextension.api.AttributesExtension;
import de.siphalor.tweed5.coat.bridge.api.TweedCoatAttributes;
import de.siphalor.tweed5.coat.bridge.api.mapping.TweedCoatEntryCreationContext;
import de.siphalor.tweed5.coat.bridge.api.mapping.TweedCoatEntryMappingContext;
import de.siphalor.tweed5.coat.bridge.api.mapping.TweedCoatEntryMappingResult;
import de.siphalor.tweed5.coat.bridge.api.mapping.TweedCoatMapper;
import de.siphalor.tweed5.coat.bridge.api.mapping.handler.BasicTweedCoatEntryHandler;
import de.siphalor.tweed5.coat.bridge.api.mapping.handler.ConvertingTweedCoatEntryHandler;
import de.siphalor.tweed5.core.api.entry.CompoundConfigEntry;
import de.siphalor.tweed5.core.api.entry.ConfigEntry;
import net.minecraft.client.Minecraft;
import net.minecraft.resources.ResourceLocation;
import org.jspecify.annotations.NonNull;
import org.jspecify.annotations.Nullable;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import static de.siphalor.tweed5.coat.bridge.api.TweedCoatMappingUtils.translatableComponentWithFallback;

@SuppressWarnings("unchecked")
public class TweedCoatMappersImpl {
	private static final org.apache.commons.logging.Log log = org.apache.commons.logging.LogFactory.getLog(TweedCoatMappersImpl.class);
	public static TweedCoatMapper<Byte> BYTE_TEXT_MAPPER = convertingTextMapper(new Class[] {Byte.class, byte.class}, value -> Byte.toString(value), Byte::parseByte);
	public static TweedCoatMapper<Short> SHORT_TEXT_MAPPER = convertingTextMapper(new Class[] {Short.class, short.class}, value -> Short.toString(value), Short::parseShort);
	public static TweedCoatMapper<Integer> INTEGER_TEXT_MAPPER = convertingTextMapper(new Class[] {Integer.class, int.class}, value -> Integer.toString(value), Integer::parseInt);
	public static TweedCoatMapper<Long> LONG_TEXT_MAPPER = convertingTextMapper(new Class[] {Long.class, long.class}, value -> Long.toString(value), Long::parseLong);
	public static TweedCoatMapper<Float> FLOAT_TEXT_MAPPER = convertingTextMapper(new Class[] {Float.class, float.class}, value -> Float.toString(value), Float::parseFloat);
	public static TweedCoatMapper<Double> DOUBLE_TEXT_MAPPER = convertingTextMapper(new Class[] {Double.class, double.class}, value -> Double.toString(value), Double::parseDouble);
	public static TweedCoatMapper<Boolean> BOOLEAN_CHECKBOX_MAPPER = new SimpleMapper<Boolean>(new Class[] {Boolean.class, boolean.class}, CheckBoxConfigInput::new);
	public static TweedCoatMapper<String> STRING_TEXT_MAPPER = new SimpleMapper<String>(new Class[] {String.class}, TextConfigInput::new);
	public static TweedCoatMapper<Enum<?>> ENUM_CYCLE_BUTTON_MAPPER = new EnumCycleButtonMapper<>();
	public static TweedCoatMapper<Object> COMPOUND_CATEGORY_MAPPER = new CompoundCategoryMapper<>();

	public static <T> TweedCoatMapper<T> convertingTextMapper(Class<T>[] valueClasses, Function<T, String> textMapper, Function<String, T> textParser) {
		return new ConvertingTextMapper<>(valueClasses, textMapper, textParser);
	}


	public static class ConvertingTextMapper<T> implements TweedCoatMapper<T> {
		private final Class<T>[] valueClasses;
		private final Function<T, String> textMapper;
		private final Function<String, T> textParser;

		@Override
		public TweedCoatEntryMappingResult<T, String> mapEntry(ConfigEntry<T> entry, TweedCoatEntryMappingContext context) {
			if (!anyClassMatches(entry.valueClass(), valueClasses)) {
				return TweedCoatEntryMappingResult.notApplicable();
			}
			return new TweedCoatEntryMappingResult<T, String>() {
				@Override
				public boolean isApplicable() {
					return true;
				}
				@Override
				public ConfigInput<String> createInput(TweedCoatEntryCreationContext<T> context) {
					return new TextConfigInput(textMapper.apply(context.currentValue()));
				}
				@Override
				public ConfigEntryHandler<String> createHandler(TweedCoatEntryCreationContext<T> context) {
					if (context.parentSaveHandler() == null) {
						throw new IllegalArgumentException("No parent save handler provided");
					}
					return new ConvertingTweedCoatEntryHandler<>(new BasicTweedCoatEntryHandler<>(context.entry(), context.defaultValue(), context.parentSaveHandler()), textMapper, textParser);
				}
				@Override
				@Nullable
				public ConfigContentWidget createContentWidget(TweedCoatEntryCreationContext<T> context) {
					return null;
				}
			};
		}

		public ConvertingTextMapper(final Class<T>[] valueClasses, final Function<T, String> textMapper, final Function<String, T> textParser) {
			this.valueClasses = valueClasses;
			this.textMapper = textMapper;
			this.textParser = textParser;
		}
	}


	public static class EnumeratedMaterialCycleButtonMapper<T> implements TweedCoatMapper<T> {
		private final Class<T> valueClass;
		private final EnumeratedMaterial<T> material;

		@Override
		public TweedCoatEntryMappingResult<T, ?> mapEntry(ConfigEntry<T> entry, TweedCoatEntryMappingContext context) {
			if (!valueClass.isAssignableFrom(entry.valueClass())) {
				return TweedCoatEntryMappingResult.notApplicable();
			}
			return new CycleButtonMappingResult<>(material);
		}

		public EnumeratedMaterialCycleButtonMapper(final Class<T> valueClass, final EnumeratedMaterial<T> material) {
			this.valueClass = valueClass;
			this.material = material;
		}
	}


	private static class EnumCycleButtonMapper<T extends Enum<?>> implements TweedCoatMapper<T> {
		@Override
		public TweedCoatEntryMappingResult<T, ?> mapEntry(ConfigEntry<T> entry, TweedCoatEntryMappingContext context) {
			if (!Enum.class.isAssignableFrom(entry.valueClass())) {
				return TweedCoatEntryMappingResult.notApplicable();
			}
			Class<T> enumClass = entry.valueClass();
			String translationKeyPrefix = entry.container().extension(AttributesExtension.class).map(extension -> extension.getAttributeValue(entry, TweedCoatAttributes.ENUM_TRANSLATION_KEY)).orElse(enumClass.getPackage().getName());
			CoatEnumMaterial<T> material = new CoatEnumMaterial<>(enumClass, translationKeyPrefix + ".");
			return new CycleButtonMappingResult<>(material);
		}
	}


	private static class CycleButtonMappingResult<T> implements TweedCoatEntryMappingResult<T, T> {
		private final EnumeratedMaterial<T> material;

		@Override
		public boolean isApplicable() {
			return true;
		}

		@Override
		public ConfigInput<T> createInput(TweedCoatEntryCreationContext<T> context) {
			return new CycleButtonConfigInput<>(material, false, context.currentValue());
		}

		@Override
		public ConfigEntryHandler<T> createHandler(TweedCoatEntryCreationContext<T> context) {
			if (context.parentSaveHandler() == null) {
				throw new IllegalArgumentException("No parent save handler provided");
			}
			return new BasicTweedCoatEntryHandler<>(context.entry(), context.defaultValue(), context.parentSaveHandler());
		}

		@Override
		@Nullable
		public ConfigContentWidget createContentWidget(TweedCoatEntryCreationContext<T> context) {
			return null;
		}

		public CycleButtonMappingResult(final EnumeratedMaterial<T> material) {
			this.material = material;
		}
	}


	private static class SimpleMapper<T> implements TweedCoatMapper<T> {
		private final Class<T>[] valueClasses;
		private final Function<T, ConfigInput<T>> inputFactory;

		@Override
		public TweedCoatEntryMappingResult<T, T> mapEntry(ConfigEntry<T> entry, TweedCoatEntryMappingContext context) {
			if (!anyClassMatches(entry.valueClass(), valueClasses)) {
				return TweedCoatEntryMappingResult.notApplicable();
			}
			return new TweedCoatEntryMappingResult<T, T>() {
				@Override
				public boolean isApplicable() {
					return true;
				}
				@Override
				public ConfigInput<T> createInput(TweedCoatEntryCreationContext<T> context) {
					return inputFactory.apply(context.currentValue());
				}
				@Override
				public ConfigEntryHandler<T> createHandler(TweedCoatEntryCreationContext<T> context) {
					if (context.parentSaveHandler() == null) {
						throw new IllegalArgumentException("No parent save handler provided");
					}
					return new BasicTweedCoatEntryHandler<>(context.entry(), context.defaultValue(), context.parentSaveHandler());
				}
				@Override
				@Nullable
				public ConfigContentWidget createContentWidget(TweedCoatEntryCreationContext<T> context) {
					return null;
				}
			};
		}

		public SimpleMapper(final Class<T>[] valueClasses, final Function<T, ConfigInput<T>> inputFactory) {
			this.valueClasses = valueClasses;
			this.inputFactory = inputFactory;
		}
	}


	private static class CompoundCategoryMapper<T> implements TweedCoatMapper<T> {
		@Override
		public TweedCoatEntryMappingResult<T, ?> mapEntry(ConfigEntry<T> entry, TweedCoatEntryMappingContext mappingContext) {

			final class MappedEntry<U> {
				private final String name;
				private final String translationKeyPrefix;
				private final ConfigEntry<U> entry;
				private final TweedCoatEntryMappingContext mappingContext;
				private final TweedCoatEntryMappingResult<U, ?> mappingResult;

				public MappedEntry(final String name, final String translationKeyPrefix, final ConfigEntry<U> entry, final TweedCoatEntryMappingContext mappingContext, final TweedCoatEntryMappingResult<U, ?> mappingResult) {
					this.name = name;
					this.translationKeyPrefix = translationKeyPrefix;
					this.entry = entry;
					this.mappingContext = mappingContext;
					this.mappingResult = mappingResult;
				}

				public String name() {
					return this.name;
				}

				public String translationKeyPrefix() {
					return this.translationKeyPrefix;
				}

				public ConfigEntry<U> entry() {
					return this.entry;
				}

				public TweedCoatEntryMappingContext mappingContext() {
					return this.mappingContext;
				}

				public TweedCoatEntryMappingResult<U, ?> mappingResult() {
					return this.mappingResult;
				}

				@Override
				public boolean equals(final Object o) {
					if (o == this) return true;
					if (!(o instanceof MappedEntry)) return false;
					final MappedEntry<?> other = (MappedEntry<?>) o;
					final Object this$name = this.name();
					final Object other$name = other.name();
					if (this$name == null ? other$name != null : !this$name.equals(other$name)) return false;
					final Object this$translationKeyPrefix = this.translationKeyPrefix();
					final Object other$translationKeyPrefix = other.translationKeyPrefix();
					if (this$translationKeyPrefix == null ? other$translationKeyPrefix != null : !this$translationKeyPrefix.equals(other$translationKeyPrefix)) return false;
					final Object this$entry = this.entry();
					final Object other$entry = other.entry();
					if (this$entry == null ? other$entry != null : !this$entry.equals(other$entry)) return false;
					final Object this$mappingContext = this.mappingContext();
					final Object other$mappingContext = other.mappingContext();
					if (this$mappingContext == null ? other$mappingContext != null : !this$mappingContext.equals(other$mappingContext)) return false;
					final Object this$mappingResult = this.mappingResult();
					final Object other$mappingResult = other.mappingResult();
					if (this$mappingResult == null ? other$mappingResult != null : !this$mappingResult.equals(other$mappingResult)) return false;
					return true;
				}

				@Override
				public int hashCode() {
					final int PRIME = 59;
					int result = 1;
					final Object $name = this.name();
					result = result * PRIME + ($name == null ? 43 : $name.hashCode());
					final Object $translationKeyPrefix = this.translationKeyPrefix();
					result = result * PRIME + ($translationKeyPrefix == null ? 43 : $translationKeyPrefix.hashCode());
					final Object $entry = this.entry();
					result = result * PRIME + ($entry == null ? 43 : $entry.hashCode());
					final Object $mappingContext = this.mappingContext();
					result = result * PRIME + ($mappingContext == null ? 43 : $mappingContext.hashCode());
					final Object $mappingResult = this.mappingResult();
					result = result * PRIME + ($mappingResult == null ? 43 : $mappingResult.hashCode());
					return result;
				}

				@Override
				public String toString() {
					return "MappedEntry(name=" + this.name() + ", translationKeyPrefix=" + this.translationKeyPrefix() + ", entry=" + this.entry() + ", mappingContext=" + this.mappingContext() + ", mappingResult=" + this.mappingResult() + ")";
				}
			}
			if (!(entry instanceof CompoundConfigEntry)) {
				return TweedCoatEntryMappingResult.notApplicable();
			}
			CompoundConfigEntry<T> compoundEntry = (CompoundConfigEntry<T>) entry;
			Optional<AttributesExtension> attributesExtension = entry.container().extension(AttributesExtension.class);
			ResourceLocation backgroundTexture = attributesExtension.map(extension -> extension.getAttributeValue(entry, TweedCoatAttributes.BACKGROUND_TEXTURE)).map(ResourceLocation::tryParse).orElse(null);
			String translationKey = attributesExtension.map(extension -> extension.getAttributeValue(entry, TweedCoatAttributes.TRANSLATION_KEY)).orElse(mappingContext.translationKeyPrefix());
			List<MappedEntry<Object>> mappedEntries = compoundEntry.subEntries().entrySet().stream().map(mapEntry -> {
				String subTranslationKeyPrefix = translationKey + "." + mapEntry.getKey();
				TweedCoatEntryMappingContext subMappingContext = mappingContext.subContextBuilder(mapEntry.getKey()).translationKeyPrefix(subTranslationKeyPrefix).parentWidgetClass(ConfigCategoryWidget.class).build();
				return new MappedEntry<>(mapEntry.getKey(), subTranslationKeyPrefix, (ConfigEntry<Object>) mapEntry.getValue(), subMappingContext, (TweedCoatEntryMappingResult<@NonNull Object, ?>) subMappingContext.mapEntry(mapEntry.getValue(), subMappingContext));
			}).filter(mappedEntry -> mappedEntry.mappingResult.isApplicable()).collect(Collectors.toList());
			return new TweedCoatEntryMappingResult<T, T>() {
				@Override
				public boolean isApplicable() {
					return true;
				}
				@Override
				@Nullable
				public ConfigInput<T> createInput(TweedCoatEntryCreationContext<T> context) {
					return null;
				}
				@Override
				@Nullable
				public ConfigEntryHandler<T> createHandler(TweedCoatEntryCreationContext<T> context) {
					return null;
				}
				@Override
				public ConfigContentWidget createContentWidget(TweedCoatEntryCreationContext<T> context) {
					ConfigCategoryWidget categoryWidget = new ConfigCategoryWidget(Minecraft.getInstance(), translatableComponentWithFallback(translationKey, mappingContext.entryName()), Collections.emptyList(), backgroundTexture);
					for (MappedEntry<Object> mappedEntry : mappedEntries) {
						TweedCoatEntryMappingResult<Object, ?> mappingResult = mappedEntry.mappingResult();
						if (!mappingResult.isApplicable()) {
							log.warn("Failed to resolve mapping for entry \"" + mappedEntry.name() + "\" at \"" + translationKey + "\". Entry will be ignored in UI.");
							continue;
						}
						Object subEntryValue = compoundEntry.get(context.currentValue(), mappedEntry.name());
						Object subEntryDefaultValue = compoundEntry.get(context.defaultValue(), mappedEntry.name());
						TweedCoatEntryCreationContext<Object> creationContext = TweedCoatEntryCreationContext.builder().entry(mappedEntry.entry()).currentValue(subEntryValue).defaultValue(subEntryDefaultValue).parentSaveHandler(value -> compoundEntry.set(context.currentValue(), mappedEntry.name(), value)).build();
						ConfigInput<?> input = mappingResult.createInput(creationContext);
						if (input != null) {
							ConfigCategoryConfigEntry<Object> entry = new ConfigCategoryConfigEntry<>(translatableComponentWithFallback(mappedEntry.translationKeyPrefix(), mappedEntry.name()), translatableComponentWithFallback(mappedEntry.translationKeyPrefix() + ".description", null), (ConfigEntryHandler<Object>) mappingResult.createHandler(creationContext), (ConfigInput<Object>) input);
							categoryWidget.addEntry(entry);
							continue;
						}
						ConfigContentWidget contentWidget = mappingResult.createContentWidget(creationContext);
						if (contentWidget != null) {
							categoryWidget.addSubTree(contentWidget);
						}
					}
					return categoryWidget;
				}
			};
		}
	}

	private static boolean anyClassMatches(Class<?> valueClass, Class<?>... classes) {
		for (Class<?> clazz : classes) {
			if (clazz.isAssignableFrom(valueClass)) {
				return true;
			}
		}
		return false;
	}
}
