package de.siphalor.tweed5.fabric.helper.api;

import com.google.gson.stream.JsonReader;
import de.siphalor.tweed5.commentloaderextension.api.CommentLoaderExtension;
import de.siphalor.tweed5.commentloaderextension.api.CommentPathProcessor;
import de.siphalor.tweed5.core.api.container.ConfigContainer;
import de.siphalor.tweed5.dataapi.api.TweedDataReader;
import de.siphalor.tweed5.data.gson.GsonReader;
import org.jspecify.annotations.Nullable;
import java.io.InputStream;
import java.io.InputStreamReader;

public class FabricConfigCommentLoader {
	private static final org.apache.commons.logging.Log log = org.apache.commons.logging.LogFactory.getLog(FabricConfigCommentLoader.class);
	private final ConfigContainer<?> configContainer;
	private final String modId;
	/**
	 * The prefix of the language keys, <b>without the trailing dot!</b>
	 */
	private final String prefix;
	/**
	 * An optional suffix of the language keys
	 */
	@Nullable
	private final String suffix;

	public void loadCommentsFromLanguageFile(String languageCode) {
		CommentLoaderExtension commentLoaderExtension = configContainer.extension(CommentLoaderExtension.class).orElseThrow(() -> new IllegalStateException("CommentLoaderExtension not declared on config"));
		String langFilePath = "assets/" + modId + "/lang/" + languageCode + ".json";
		InputStream langInputStream = getClass().getClassLoader().getResourceAsStream(langFilePath);
		if (langInputStream == null) {
			log.warn("Failed to find language file " + langFilePath + " for loading config comments");
			return;
		}
		try (TweedDataReader reader = new GsonReader(new JsonReader(new InputStreamReader(langInputStream)))) {
			commentLoaderExtension.loadComments(reader, new CommentPathProcessor() {
				@Override
				public MatchStatus matches(String path) {
					if (!path.startsWith(prefix)) {
						return MatchStatus.NO;
					} else if (path.length() == prefix.length()) {
						if (suffix != null && !path.endsWith(suffix)) {
							return MatchStatus.NO;
						}
						return MatchStatus.YES;
					} else if (path.charAt(prefix.length()) != '.') {
						return MatchStatus.NO;
					} else if (suffix != null && !path.endsWith(suffix)) {
						return MatchStatus.MAYBE_DEEPER;
					} else {
						return MatchStatus.YES;
					}
				}
				@Override
				public String process(String path) {
					if (path.equals(prefix)) {
						return "";
					}
					path = path.substring(prefix.length() + 1);
					if (suffix != null) {
						path = path.substring(0, path.length() - suffix.length());
					}
					return path;
				}
			});
		} catch (Exception e) {
			log.warn("Failed to load comments from language file", e);
		}
	}

	/**
	 * Creates a new {@code FabricConfigCommentLoader} instance.
	 *
	 * @param configContainer
	 * @param modId
	 * @param prefix The prefix of the language keys, <b>without the trailing dot!</b>
	 * @param suffix An optional suffix of the language keys
	 */
	FabricConfigCommentLoader(final ConfigContainer<?> configContainer, final String modId, final String prefix, @Nullable final String suffix) {
		this.configContainer = configContainer;
		this.modId = modId;
		this.prefix = prefix;
		this.suffix = suffix;
	}


	public static class FabricConfigCommentLoaderBuilder {
		private ConfigContainer<?> configContainer;
		private String modId;
		private String prefix;
		private String suffix;

		FabricConfigCommentLoaderBuilder() {
		}

		/**
		 * @return {@code this}.
		 */
		public FabricConfigCommentLoader.FabricConfigCommentLoaderBuilder configContainer(final ConfigContainer<?> configContainer) {
			this.configContainer = configContainer;
			return this;
		}

		/**
		 * @return {@code this}.
		 */
		public FabricConfigCommentLoader.FabricConfigCommentLoaderBuilder modId(final String modId) {
			this.modId = modId;
			return this;
		}

		/**
		 * The prefix of the language keys, <b>without the trailing dot!</b>
		 * @return {@code this}.
		 */
		public FabricConfigCommentLoader.FabricConfigCommentLoaderBuilder prefix(final String prefix) {
			this.prefix = prefix;
			return this;
		}

		/**
		 * An optional suffix of the language keys
		 * @return {@code this}.
		 */
		public FabricConfigCommentLoader.FabricConfigCommentLoaderBuilder suffix(@Nullable final String suffix) {
			this.suffix = suffix;
			return this;
		}

		public FabricConfigCommentLoader build() {
			return new FabricConfigCommentLoader(this.configContainer, this.modId, this.prefix, this.suffix);
		}

		@Override
		public String toString() {
			return "FabricConfigCommentLoader.FabricConfigCommentLoaderBuilder(configContainer=" + this.configContainer + ", modId=" + this.modId + ", prefix=" + this.prefix + ", suffix=" + this.suffix + ")";
		}
	}

	public static FabricConfigCommentLoader.FabricConfigCommentLoaderBuilder builder() {
		return new FabricConfigCommentLoader.FabricConfigCommentLoaderBuilder();
	}
}
