package de.siphalor.tweed5.weaver.pojoext.serde.api.auto;

import de.siphalor.tweed5.core.api.container.ConfigContainer;
import de.siphalor.tweed5.core.api.entry.ConfigEntry;
import de.siphalor.tweed5.core.api.entry.NullableConfigEntry;
import de.siphalor.tweed5.data.extension.api.ReadWriteExtension;
import de.siphalor.tweed5.data.extension.api.TweedEntryReader;
import de.siphalor.tweed5.data.extension.api.TweedEntryWriter;
import de.siphalor.tweed5.data.extension.api.TweedReaderWriterProvider;
import de.siphalor.tweed5.data.extension.api.readwrite.TweedEntryReaderWriters;
import de.siphalor.tweed5.data.extension.impl.TweedEntryReaderWriterImpls;
import de.siphalor.tweed5.patchwork.api.Patchwork;
import de.siphalor.tweed5.patchwork.api.PatchworkPartAccess;
import de.siphalor.tweed5.typeutils.api.type.ActualType;
import de.siphalor.tweed5.weaver.pojo.api.weaving.ProtoWeavingContext;
import de.siphalor.tweed5.weaver.pojo.api.weaving.TweedPojoWeavingExtension;
import de.siphalor.tweed5.weaver.pojo.api.weaving.WeavingContext;
import de.siphalor.tweed5.weaver.pojoext.serde.impl.ReaderWriterLoader;
import de.siphalor.tweed5.weaver.pojoext.serde.impl.SerdePojoReaderWriterSpec;
import org.jetbrains.annotations.ApiStatus;
import org.jspecify.annotations.Nullable;
import java.util.*;

public class AutoReadWritePojoWeavingProcessor implements TweedPojoWeavingExtension {
	private final ReadWriteExtension readWriteExtension;
	private final ReaderWriterLoader readerWriterLoader = new ReaderWriterLoader();
	@Nullable
	private PatchworkPartAccess<CustomData> customDataAccess;

	@ApiStatus.Internal
	public AutoReadWritePojoWeavingProcessor(ConfigContainer<?> configContainer) {
		this.readWriteExtension = configContainer.extension(ReadWriteExtension.class).orElseThrow(() -> new IllegalStateException("You must register a " + ReadWriteExtension.class.getSimpleName() + " to use the " + getClass().getSimpleName()));
	}

	@Override
	public void setup(SetupContext context) {
		customDataAccess = context.registerWeavingContextExtensionData(CustomData.class);
		loadProviders();
	}

	private void loadProviders() {
		ServiceLoader<TweedReaderWriterProvider> serviceLoader = ServiceLoader.load(TweedReaderWriterProvider.class);
		serviceLoader.forEach(readerWriterLoader::load);
	}

	@Override
	public <T> void beforeWeaveEntry(ActualType<T> valueType, Patchwork extensionsData, ProtoWeavingContext context) {
		assert customDataAccess != null;
		CustomData existingCustomData = extensionsData.get(customDataAccess);
		List<Mapping> existingMappings = existingCustomData == null ? Collections.emptyList() : existingCustomData.mappings();
		AutoReadWriteMapping[] mappingAnnotations = context.annotations().getAnnotationsByType(AutoReadWriteMapping.class);
		if (existingCustomData == null || mappingAnnotations.length > 0) {
			List<Mapping> mappings;
			if (existingMappings.isEmpty() && mappingAnnotations.length == 0) {
				mappings = Collections.emptyList();
			} else {
				mappings = new ArrayList<>(existingMappings.size() + mappingAnnotations.length + 5);
				mappings.addAll(existingMappings);
				for (AutoReadWriteMapping mappingAnnotation : mappingAnnotations) {
					mappings.add(annotationToMapping(mappingAnnotation));
				}
			}
			extensionsData.set(customDataAccess, new CustomData(mappings));
		}
	}

	private Mapping annotationToMapping(AutoReadWriteMapping annotation) {
		return new Mapping(annotation.entryClasses(), annotation.valueClasses(), resolveReader(annotation.reader().isEmpty() ? annotation.spec() : annotation.reader()), resolveWriter(annotation.writer().isEmpty() ? annotation.spec() : annotation.writer()));
	}

	private TweedEntryReader<?, ?> resolveReader(String specText) {
		if (specText.isEmpty()) {
			return TweedEntryReaderWriterImpls.NOOP_READER_WRITER;
		}
		try {
			SerdePojoReaderWriterSpec spec = SerdePojoReaderWriterSpec.parse(specText);
			return readerWriterLoader.resolveReaderFromSpec(spec);
		} catch (Exception e) {
			throw new IllegalArgumentException("Failed to parse definition for reader: \"" + specText + "\"", e);
		}
	}

	private TweedEntryWriter<?, ?> resolveWriter(String specText) {
		if (specText.isEmpty()) {
			return TweedEntryReaderWriterImpls.NOOP_READER_WRITER;
		}
		try {
			SerdePojoReaderWriterSpec spec = SerdePojoReaderWriterSpec.parse(specText);
			return readerWriterLoader.resolveWriterFromSpec(spec);
		} catch (Exception e) {
			throw new IllegalArgumentException("Failed to parse definition for writer: \"" + specText + "\"", e);
		}
	}

	@Override
	public <T> void afterWeaveEntry(ActualType<T> valueType, ConfigEntry<T> configEntry, WeavingContext context) {
		if (configEntry instanceof NullableConfigEntry) {
			readWriteExtension.setEntryReaderWriter(configEntry, TweedEntryReaderWriters.nullableReaderWriter(), TweedEntryReaderWriters.nullableReaderWriter());
			return;
		}
		assert customDataAccess != null;
		CustomData customData = context.extensionsData().get(customDataAccess);
		if (customData == null || customData.mappings().isEmpty()) {
			return;
		}
		Mapping mapping = determineMapping(customData, configEntry, valueType);
		if (mapping == null) {
			return;
		}
		//noinspection unchecked
		readWriteExtension.setEntryReaderWriter((ConfigEntry<Object>) configEntry, (TweedEntryReader<Object, @org.jspecify.annotations.NonNull ConfigEntry<Object>>) mapping.reader(), (TweedEntryWriter<Object, @org.jspecify.annotations.NonNull ConfigEntry<Object>>) mapping.writer());
	}

	@Nullable
	private Mapping determineMapping(CustomData customData, ConfigEntry<?> configEntry, ActualType<?> valueType) {
		Mapping strictMapping = customData.strictMappings().get(new MappingStrictKey(configEntry.getClass(), valueType.declaredType()));
		if (strictMapping != null) {
			return strictMapping;
		}
		for (int i = customData.mappings().size() - 1; i >= 0; i--) {
			Mapping mapping = customData.mappings().get(i);
			if (mappingMatches(mapping, configEntry, valueType)) {
				customData.strictMappings.put(new MappingStrictKey(configEntry.getClass(), valueType.declaredType()), mapping);
				return mapping;
			}
		}
		return null;
	}

	private boolean mappingMatches(Mapping mapping, ConfigEntry<?> configEntry, ActualType<?> valueType) {
		return anyClassMatches(mapping.entryClasses, configEntry.getClass()) && anyClassMatches(mapping.valueClasses, valueType.declaredType());
	}

	private boolean anyClassMatches(Class<?>[] haystack, Class<?> needle) {
		for (Class<?> hay : haystack) {
			if (hay.isAssignableFrom(needle)) {
				return true;
			}
		}
		return false;
	}


	private static final class CustomData {
		private final List<Mapping> mappings;
		private final Map<MappingStrictKey, Mapping> strictMappings = new HashMap<>();

		public CustomData(final List<Mapping> mappings) {
			this.mappings = mappings;
		}

		public List<Mapping> mappings() {
			return this.mappings;
		}

		public Map<MappingStrictKey, Mapping> strictMappings() {
			return this.strictMappings;
		}

		@Override
		public boolean equals(final Object o) {
			if (o == this) return true;
			if (!(o instanceof AutoReadWritePojoWeavingProcessor.CustomData)) return false;
			final AutoReadWritePojoWeavingProcessor.CustomData other = (AutoReadWritePojoWeavingProcessor.CustomData) o;
			final Object this$mappings = this.mappings();
			final Object other$mappings = other.mappings();
			if (this$mappings == null ? other$mappings != null : !this$mappings.equals(other$mappings)) return false;
			final Object this$strictMappings = this.strictMappings();
			final Object other$strictMappings = other.strictMappings();
			if (this$strictMappings == null ? other$strictMappings != null : !this$strictMappings.equals(other$strictMappings)) return false;
			return true;
		}

		@Override
		public int hashCode() {
			final int PRIME = 59;
			int result = 1;
			final Object $mappings = this.mappings();
			result = result * PRIME + ($mappings == null ? 43 : $mappings.hashCode());
			final Object $strictMappings = this.strictMappings();
			result = result * PRIME + ($strictMappings == null ? 43 : $strictMappings.hashCode());
			return result;
		}

		@Override
		public String toString() {
			return "AutoReadWritePojoWeavingProcessor.CustomData(mappings=" + this.mappings() + ", strictMappings=" + this.strictMappings() + ")";
		}
	}


	private static final class Mapping {
		private final Class<?>[] entryClasses;
		private final Class<?>[] valueClasses;
		private final TweedEntryReader<?, ?> reader;
		private final TweedEntryWriter<?, ?> writer;

		public Mapping(final Class<?>[] entryClasses, final Class<?>[] valueClasses, final TweedEntryReader<?, ?> reader, final TweedEntryWriter<?, ?> writer) {
			this.entryClasses = entryClasses;
			this.valueClasses = valueClasses;
			this.reader = reader;
			this.writer = writer;
		}

		public Class<?>[] entryClasses() {
			return this.entryClasses;
		}

		public Class<?>[] valueClasses() {
			return this.valueClasses;
		}

		public TweedEntryReader<?, ?> reader() {
			return this.reader;
		}

		public TweedEntryWriter<?, ?> writer() {
			return this.writer;
		}

		@Override
		public boolean equals(final Object o) {
			if (o == this) return true;
			if (!(o instanceof AutoReadWritePojoWeavingProcessor.Mapping)) return false;
			final AutoReadWritePojoWeavingProcessor.Mapping other = (AutoReadWritePojoWeavingProcessor.Mapping) o;
			if (!java.util.Arrays.deepEquals(this.entryClasses(), other.entryClasses())) return false;
			if (!java.util.Arrays.deepEquals(this.valueClasses(), other.valueClasses())) return false;
			final Object this$reader = this.reader();
			final Object other$reader = other.reader();
			if (this$reader == null ? other$reader != null : !this$reader.equals(other$reader)) return false;
			final Object this$writer = this.writer();
			final Object other$writer = other.writer();
			if (this$writer == null ? other$writer != null : !this$writer.equals(other$writer)) return false;
			return true;
		}

		@Override
		public int hashCode() {
			final int PRIME = 59;
			int result = 1;
			result = result * PRIME + java.util.Arrays.deepHashCode(this.entryClasses());
			result = result * PRIME + java.util.Arrays.deepHashCode(this.valueClasses());
			final Object $reader = this.reader();
			result = result * PRIME + ($reader == null ? 43 : $reader.hashCode());
			final Object $writer = this.writer();
			result = result * PRIME + ($writer == null ? 43 : $writer.hashCode());
			return result;
		}

		@Override
		public String toString() {
			return "AutoReadWritePojoWeavingProcessor.Mapping(entryClasses=" + java.util.Arrays.deepToString(this.entryClasses()) + ", valueClasses=" + java.util.Arrays.deepToString(this.valueClasses()) + ", reader=" + this.reader() + ", writer=" + this.writer() + ")";
		}
	}


	private static final class MappingStrictKey {
		private final Class<?> entryClass;
		private final Class<?> valueClass;

		public MappingStrictKey(final Class<?> entryClass, final Class<?> valueClass) {
			this.entryClass = entryClass;
			this.valueClass = valueClass;
		}

		public Class<?> entryClass() {
			return this.entryClass;
		}

		public Class<?> valueClass() {
			return this.valueClass;
		}

		@Override
		public boolean equals(final Object o) {
			if (o == this) return true;
			if (!(o instanceof AutoReadWritePojoWeavingProcessor.MappingStrictKey)) return false;
			final AutoReadWritePojoWeavingProcessor.MappingStrictKey other = (AutoReadWritePojoWeavingProcessor.MappingStrictKey) o;
			final Object this$entryClass = this.entryClass();
			final Object other$entryClass = other.entryClass();
			if (this$entryClass == null ? other$entryClass != null : !this$entryClass.equals(other$entryClass)) return false;
			final Object this$valueClass = this.valueClass();
			final Object other$valueClass = other.valueClass();
			if (this$valueClass == null ? other$valueClass != null : !this$valueClass.equals(other$valueClass)) return false;
			return true;
		}

		@Override
		public int hashCode() {
			final int PRIME = 59;
			int result = 1;
			final Object $entryClass = this.entryClass();
			result = result * PRIME + ($entryClass == null ? 43 : $entryClass.hashCode());
			final Object $valueClass = this.valueClass();
			result = result * PRIME + ($valueClass == null ? 43 : $valueClass.hashCode());
			return result;
		}

		@Override
		public String toString() {
			return "AutoReadWritePojoWeavingProcessor.MappingStrictKey(entryClass=" + this.entryClass() + ", valueClass=" + this.valueClass() + ")";
		}
	}
}
