package de.siphalor.tweed5.weaver.pojo.impl.weaving;

import org.jspecify.annotations.Nullable;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.lang.reflect.Type;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;

public class PojoClassIntrospector {
	private static final org.apache.commons.logging.Log log = org.apache.commons.logging.LogFactory.getLog(PojoClassIntrospector.class);
	private final Class<?> clazz;
	private final MethodHandles.Lookup lookup = MethodHandles.publicLookup();
	@Nullable
	private Map<String, Property> properties;

	public static PojoClassIntrospector forClass(Class<?> clazz) {
		if ((clazz.getModifiers() & Modifier.PUBLIC) == 0) {
			throw new IllegalStateException("Class " + clazz.getName() + " must be public");
		}
		return new PojoClassIntrospector(clazz);
	}

	public Class<?> type() {
		return clazz;
	}

	@Nullable
	public MethodHandle noArgsConstructor() {
		try {
			return lookup.findConstructor(clazz, MethodType.methodType(void.class));
		} catch (NoSuchMethodException | IllegalAccessException | SecurityException e) {
			return null;
		}
	}

	public Map<String, Property> properties() {
		if (this.properties == null) {
			this.properties = new LinkedHashMap<>();
			Class<?> currentClass = clazz;
			while (currentClass != null) {
				appendClassProperties(currentClass);
				currentClass = currentClass.getSuperclass();
			}
		}
		return Collections.unmodifiableMap(this.properties);
	}

	private void appendClassProperties(Class<?> targetClass) {
		try {
			Field[] fields = targetClass.getDeclaredFields();
			for (Field field : fields) {
				if (shouldIgnoreField(field)) {
					continue;
				}
				if (!properties.containsKey(field.getName())) {
					Property property = introspectProperty(field);
					properties.put(property.field.getName(), property);
				} else {
					Property existingProperty = properties.get(field.getName());
					log.error("Duplicate property \"" + field.getName() + "\" detected in hierarchy of " + clazz.getName() + " in classes: " + existingProperty.field().getDeclaringClass().getName() + " and " + targetClass.getName());
				}
			}
		} catch (Exception e) {
			log.error("Got unexpected error introspecting the properties of class " + targetClass.getName() + " (in hierarchy of " + clazz.getName() + ")", e);
		}
	}

	private boolean shouldIgnoreField(Field field) {
		return (field.getModifiers() & (Modifier.STATIC | Modifier.TRANSIENT)) != 0;
	}

	private Property introspectProperty(Field field) {
		int modifiers = field.getModifiers();
		return Property.builder().field(field).isFinal((modifiers & Modifier.FINAL) != 0).getter(findGetter(field)).setter(findSetter(field)).type(field.getGenericType()).build();
	}

	@Nullable
	private MethodHandle findGetter(Field field) {
		String fieldName = field.getName();
		// fluid getters
		MethodHandle method = findMethod(clazz, new MethodDescriptor(fieldName, MethodType.methodType(field.getType())));
		if (method != null) {
			return method;
		}
		// boolean getters
		if (field.getType() == Boolean.class || field.getType() == Boolean.TYPE) {
			method = findMethod(clazz, new MethodDescriptor("is" + firstToUpper(fieldName), MethodType.methodType(field.getType())));
			if (method != null) {
				return method;
			}
		}
		// classic getters
		method = findMethod(clazz, new MethodDescriptor("get" + firstToUpper(fieldName), MethodType.methodType(field.getType())));
		if (method != null) {
			return method;
		}
		// public field access
		int modifiers = field.getModifiers();
		if ((modifiers & Modifier.PUBLIC) != 0) {
			return findFieldGetter(field);
		}
		return null;
	}

	@Nullable
	private MethodHandle findSetter(Field field) {
		String fieldName = field.getName();
		String classicSetterName = "set" + firstToUpper(fieldName);
		MethodHandle method = findFirstMethod(clazz, 
		// fluid
		new MethodDescriptor(fieldName, MethodType.methodType(Void.TYPE, field.getType())), 
		// fluid + chain
		new MethodDescriptor(fieldName, MethodType.methodType(field.getDeclaringClass(), field.getType())), 
		// classic
		new MethodDescriptor(classicSetterName, MethodType.methodType(Void.TYPE, field.getType())), 
		// classic + chain
		new MethodDescriptor(classicSetterName, MethodType.methodType(field.getDeclaringClass(), field.getType())));
		if (method != null) {
			return method;
		}
		// public field access
		int modifiers = field.getModifiers();
		if ((modifiers & Modifier.PUBLIC) != 0) {
			return findFieldSetter(field);
		}
		return null;
	}

	@Nullable
	private MethodHandle findFirstMethod(Class<?> targetClass, MethodDescriptor... methodDescriptors) {
		for (MethodDescriptor methodDescriptor : methodDescriptors) {
			MethodHandle method = findMethod(targetClass, methodDescriptor);
			if (method != null) {
				return method;
			}
		}
		return null;
	}

	@Nullable
	private MethodHandle findMethod(Class<?> targetClass, MethodDescriptor methodDescriptor) {
		try {
			return lookup.findVirtual(targetClass, methodDescriptor.name(), methodDescriptor.methodType());
		} catch (NoSuchMethodException e) {
			return null;
		} catch (IllegalAccessException e) {
			log.warn("Failed to access method \"" + methodDescriptor + "\" of class " + targetClass.getName() + " in hierarchy of " + clazz.getName(), e);
			return null;
		}
	}

	@Nullable
	private MethodHandle findFieldGetter(Field field) {
		try {
			return lookup.findGetter(field.getDeclaringClass(), field.getName(), field.getType());
		} catch (NoSuchFieldException e) {
			return null;
		} catch (IllegalAccessException e) {
			log.warn("Failed to access getter for field \"" + field.getName() + "\" of class " + field.getDeclaringClass().getName() + " in hierarchy of " + clazz.getName(), e);
			return null;
		}
	}

	@Nullable
	private MethodHandle findFieldSetter(Field field) {
		try {
			return lookup.findSetter(field.getDeclaringClass(), field.getName(), field.getType());
		} catch (NoSuchFieldException e) {
			return null;
		} catch (IllegalAccessException e) {
			log.warn("Failed to access setter for field \"" + field.getName() + "\" of class " + field.getDeclaringClass().getName() + " in hierarchy of " + clazz.getName(), e);
			return null;
		}
	}

	private static String firstToUpper(String text) {
		return Character.toUpperCase(text.charAt(0)) + text.substring(1);
	}


	private static final class MethodDescriptor {
		private final String name;
		private final MethodType methodType;

		public MethodDescriptor(final String name, final MethodType methodType) {
			this.name = name;
			this.methodType = methodType;
		}

		public String name() {
			return this.name;
		}

		public MethodType methodType() {
			return this.methodType;
		}

		@Override
		public boolean equals(final Object o) {
			if (o == this) return true;
			if (!(o instanceof PojoClassIntrospector.MethodDescriptor)) return false;
			final PojoClassIntrospector.MethodDescriptor other = (PojoClassIntrospector.MethodDescriptor) o;
			final Object this$name = this.name();
			final Object other$name = other.name();
			if (this$name == null ? other$name != null : !this$name.equals(other$name)) return false;
			final Object this$methodType = this.methodType();
			final Object other$methodType = other.methodType();
			if (this$methodType == null ? other$methodType != null : !this$methodType.equals(other$methodType)) return false;
			return true;
		}

		@Override
		public int hashCode() {
			final int PRIME = 59;
			int result = 1;
			final Object $name = this.name();
			result = result * PRIME + ($name == null ? 43 : $name.hashCode());
			final Object $methodType = this.methodType();
			result = result * PRIME + ($methodType == null ? 43 : $methodType.hashCode());
			return result;
		}

		@Override
		public String toString() {
			return "PojoClassIntrospector.MethodDescriptor(name=" + this.name() + ", methodType=" + this.methodType() + ")";
		}
	}


	public static final class Property {
		private final Field field;
		private final boolean isFinal;
		private final Type type;
		@Nullable
		private final MethodHandle getter;
		@Nullable
		private final MethodHandle setter;

		Property(final Field field, final boolean isFinal, final Type type, @Nullable final MethodHandle getter, @Nullable final MethodHandle setter) {
			this.field = field;
			this.isFinal = isFinal;
			this.type = type;
			this.getter = getter;
			this.setter = setter;
		}


		public static class PropertyBuilder {
			private Field field;
			private boolean isFinal;
			private Type type;
			private MethodHandle getter;
			private MethodHandle setter;

			PropertyBuilder() {
			}

			/**
			 * @return {@code this}.
			 */
			public PojoClassIntrospector.Property.PropertyBuilder field(final Field field) {
				this.field = field;
				return this;
			}

			/**
			 * @return {@code this}.
			 */
			public PojoClassIntrospector.Property.PropertyBuilder isFinal(final boolean isFinal) {
				this.isFinal = isFinal;
				return this;
			}

			/**
			 * @return {@code this}.
			 */
			public PojoClassIntrospector.Property.PropertyBuilder type(final Type type) {
				this.type = type;
				return this;
			}

			/**
			 * @return {@code this}.
			 */
			public PojoClassIntrospector.Property.PropertyBuilder getter(@Nullable final MethodHandle getter) {
				this.getter = getter;
				return this;
			}

			/**
			 * @return {@code this}.
			 */
			public PojoClassIntrospector.Property.PropertyBuilder setter(@Nullable final MethodHandle setter) {
				this.setter = setter;
				return this;
			}

			public PojoClassIntrospector.Property build() {
				return new PojoClassIntrospector.Property(this.field, this.isFinal, this.type, this.getter, this.setter);
			}

			@Override
			public String toString() {
				return "PojoClassIntrospector.Property.PropertyBuilder(field=" + this.field + ", isFinal=" + this.isFinal + ", type=" + this.type + ", getter=" + this.getter + ", setter=" + this.setter + ")";
			}
		}

		public static PojoClassIntrospector.Property.PropertyBuilder builder() {
			return new PojoClassIntrospector.Property.PropertyBuilder();
		}

		public Field field() {
			return this.field;
		}

		public boolean isFinal() {
			return this.isFinal;
		}

		public Type type() {
			return this.type;
		}

		@Nullable
		public MethodHandle getter() {
			return this.getter;
		}

		@Nullable
		public MethodHandle setter() {
			return this.setter;
		}

		@Override
		public boolean equals(final Object o) {
			if (o == this) return true;
			if (!(o instanceof PojoClassIntrospector.Property)) return false;
			final PojoClassIntrospector.Property other = (PojoClassIntrospector.Property) o;
			if (this.isFinal() != other.isFinal()) return false;
			final Object this$field = this.field();
			final Object other$field = other.field();
			if (this$field == null ? other$field != null : !this$field.equals(other$field)) return false;
			final Object this$type = this.type();
			final Object other$type = other.type();
			if (this$type == null ? other$type != null : !this$type.equals(other$type)) return false;
			final Object this$getter = this.getter();
			final Object other$getter = other.getter();
			if (this$getter == null ? other$getter != null : !this$getter.equals(other$getter)) return false;
			final Object this$setter = this.setter();
			final Object other$setter = other.setter();
			if (this$setter == null ? other$setter != null : !this$setter.equals(other$setter)) return false;
			return true;
		}

		@Override
		public int hashCode() {
			final int PRIME = 59;
			int result = 1;
			result = result * PRIME + (this.isFinal() ? 79 : 97);
			final Object $field = this.field();
			result = result * PRIME + ($field == null ? 43 : $field.hashCode());
			final Object $type = this.type();
			result = result * PRIME + ($type == null ? 43 : $type.hashCode());
			final Object $getter = this.getter();
			result = result * PRIME + ($getter == null ? 43 : $getter.hashCode());
			final Object $setter = this.setter();
			result = result * PRIME + ($setter == null ? 43 : $setter.hashCode());
			return result;
		}

		@Override
		public String toString() {
			return "PojoClassIntrospector.Property(field=" + this.field() + ", isFinal=" + this.isFinal() + ", type=" + this.type() + ", getter=" + this.getter() + ", setter=" + this.setter() + ")";
		}
	}

	private PojoClassIntrospector(final Class<?> clazz) {
		this.clazz = clazz;
	}
}
