package de.siphalor.tweed5.annotationinheritance.api;

import de.siphalor.tweed5.annotationinheritance.impl.AnnotationInheritanceResolver;
import de.siphalor.tweed5.typeutils.api.annotations.AnnotationRepeatType;
import de.siphalor.tweed5.utils.api.collection.ClassToInstanceMap;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jspecify.annotations.NonNull;
import java.lang.annotation.Annotation;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Array;

public class AnnotationInheritanceAwareAnnotatedElement implements AnnotatedElement {
	private final AnnotatedElement inner;
	@Nullable
	private ClassToInstanceMap<Annotation> resolvedAnnotations;

	@Override
	@Nullable
	public <T extends Annotation> T getAnnotation(@NonNull Class<T> annotationClass) {
		if (resolvedAnnotations != null) {
			return resolvedAnnotations.get(annotationClass);
		}
		Annotation[] annotations = inner.getAnnotations();
		boolean metaEncountered = false;
		T foundAnnotation = null;
		for (Annotation annotation : annotations) {
			if (annotation.annotationType().equals(annotationClass)) {
				//noinspection unchecked
				foundAnnotation = (T) annotation;
			} else if (!metaEncountered) {
				metaEncountered = annotation.annotationType().isAnnotationPresent(AnnotationInheritance.class);
			}
		}
		if (foundAnnotation != null) {
			AnnotationRepeatType repeatType = AnnotationRepeatType.getType(annotationClass);
			if (repeatType instanceof AnnotationRepeatType.NonRepeatable) {
				return foundAnnotation;
			}
		}
		if (!metaEncountered) {
			return foundAnnotation;
		}
		return getOrResolveAnnotations().get(annotationClass);
	}

	@Override
	@NonNull
	@NotNull
	public <T extends Annotation> T[] getAnnotationsByType(@NonNull Class<T> annotationClass) {
		T annotation = getOrResolveAnnotations().get(annotationClass);
		if (annotation != null) {
			//noinspection unchecked
			T[] array = (T[]) Array.newInstance(annotationClass, 1);
			array[0] = annotation;
			return array;
		}
		AnnotationRepeatType repeatType = AnnotationRepeatType.getType(annotationClass);
		if (repeatType instanceof AnnotationRepeatType.Repeatable) {
			AnnotationRepeatType.RepeatableContainer containerRepeatType = ((AnnotationRepeatType.Repeatable) repeatType).containerRepeatType();
			Annotation containerAnnotation = getOrResolveAnnotations().get(containerRepeatType.annotationClass());
			if (containerAnnotation != null) {
				return containerRepeatType.elements(containerAnnotation);
			}
		}
		//noinspection unchecked
		return (T[]) Array.newInstance(annotationClass, 0);
	}

	@Override
	@NotNull
	public Annotation[] getAnnotations() {
		return getOrResolveAnnotations().values().toArray(new Annotation[0]);
	}

	@Override
	@NotNull
	public Annotation[] getDeclaredAnnotations() {
		return inner.getDeclaredAnnotations();
	}

	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append(getClass().getSimpleName()).append('[');
		getOrResolveAnnotations().forEach(annotation -> sb.append(annotation).append(';'));
		return sb.append(']').toString();
	}

	private ClassToInstanceMap<Annotation> getOrResolveAnnotations() {
		if (resolvedAnnotations == null) {
			resolvedAnnotations = new AnnotationInheritanceResolver(inner).resolve();
		}
		return resolvedAnnotations;
	}

	public AnnotationInheritanceAwareAnnotatedElement(final AnnotatedElement inner) {
		this.inner = inner;
	}
}
