package de.siphalor.tweed5.defaultextensions.comment.impl;

import com.google.auto.service.AutoService;
import de.siphalor.tweed5.core.api.container.ConfigContainer;
import de.siphalor.tweed5.core.api.container.ConfigContainerSetupPhase;
import de.siphalor.tweed5.core.api.entry.ConfigEntry;
import de.siphalor.tweed5.core.api.extension.TweedExtension;
import de.siphalor.tweed5.core.api.extension.TweedExtensionSetupContext;
import de.siphalor.tweed5.core.api.middleware.DefaultMiddlewareContainer;
import de.siphalor.tweed5.data.extension.api.extension.ReadWriteExtensionSetupContext;
import de.siphalor.tweed5.data.extension.api.extension.ReadWriteRelatedExtension;
import de.siphalor.tweed5.defaultextensions.comment.api.CommentExtension;
import de.siphalor.tweed5.defaultextensions.comment.api.CommentModifyingExtension;
import de.siphalor.tweed5.defaultextensions.comment.api.CommentProducer;
import de.siphalor.tweed5.patchwork.api.PatchworkPartAccess;
import org.jspecify.annotations.Nullable;

@AutoService(CommentExtension.class)
public class CommentExtensionImpl implements ReadWriteRelatedExtension, CommentExtension {
	private final ConfigContainer<?> configContainer;
	private final PatchworkPartAccess<CustomEntryData> customEntryDataAccess;
	private final DefaultMiddlewareContainer<CommentProducer> middlewareContainer;
	@Nullable
	private PatchworkPartAccess<Boolean> writerInstalledReadWriteContextAccess;

	public CommentExtensionImpl(ConfigContainer<?> configContainer, TweedExtensionSetupContext context) {
		this.configContainer = configContainer;
		this.customEntryDataAccess = context.registerEntryExtensionData(CustomEntryData.class);
		this.middlewareContainer = new DefaultMiddlewareContainer<>();
	}

	@Override
	public String getId() {
		return EXTENSION_ID;
	}

	@Override
	public void extensionsFinalized() {
		for (TweedExtension extension : configContainer.extensions()) {
			if (extension instanceof CommentModifyingExtension) {
				middlewareContainer.register(((CommentModifyingExtension) extension).commentMiddleware());
			}
		}
		middlewareContainer.seal();
	}

	@Override
	public void setupReadWriteExtension(ReadWriteExtensionSetupContext context) {
		writerInstalledReadWriteContextAccess = context.registerReadWriteContextExtensionData(Boolean.class);
		context.registerWriterMiddleware(new TweedEntryWriterCommentMiddleware(this));
	}

	@Override
	public void setBaseComment(ConfigEntry<?> configEntry, String baseComment) {
		if (configEntry.container() != configContainer) {
			throw new IllegalArgumentException("config entry doesn\'t belong to config container of this extension");
		} else if (configContainer.setupPhase().compareTo(ConfigContainerSetupPhase.INITIALIZED) >= 0) {
			throw new IllegalStateException("config container must not be initialized");
		}
		getOrCreateCustomEntryData(configEntry).baseComment(baseComment);
	}

	@Override
	public void initialize() {
		recomputeFullComments();
	}

	@Override
	public void recomputeFullComments() {
		configContainer.rootEntry().visitInOrder(entry -> {
			CustomEntryData entryData = getOrCreateCustomEntryData(entry);
			entryData.commentProducer(middlewareContainer.process(_entry -> entryData.baseComment()));
		});
	}

	private CustomEntryData getOrCreateCustomEntryData(ConfigEntry<?> entry) {
		CustomEntryData customEntryData = entry.extensionsData().get(customEntryDataAccess);
		if (customEntryData == null) {
			customEntryData = new CustomEntryData();
			entry.extensionsData().set(customEntryDataAccess, customEntryData);
		}
		return customEntryData;
	}

	@Override
	@Nullable
	public String getFullComment(ConfigEntry<?> configEntry) {
		CustomEntryData customEntryData = configEntry.extensionsData().get(customEntryDataAccess);
		if (customEntryData == null) {
			return null;
		}
		String comment = customEntryData.commentProducer().createComment(configEntry);
		return comment.isEmpty() ? null : comment;
	}


	private static class CustomEntryData {
		private String baseComment = "";
		private CommentProducer commentProducer;

		public CustomEntryData() {
		}

		public String baseComment() {
			return this.baseComment;
		}

		public CommentProducer commentProducer() {
			return this.commentProducer;
		}

		/**
		 * @return {@code this}.
		 */
		public CommentExtensionImpl.CustomEntryData baseComment(final String baseComment) {
			this.baseComment = baseComment;
			return this;
		}

		/**
		 * @return {@code this}.
		 */
		public CommentExtensionImpl.CustomEntryData commentProducer(final CommentProducer commentProducer) {
			this.commentProducer = commentProducer;
			return this;
		}

		@Override
		public boolean equals(final Object o) {
			if (o == this) return true;
			if (!(o instanceof CommentExtensionImpl.CustomEntryData)) return false;
			final CommentExtensionImpl.CustomEntryData other = (CommentExtensionImpl.CustomEntryData) o;
			if (!other.canEqual((Object) this)) return false;
			final Object this$baseComment = this.baseComment();
			final Object other$baseComment = other.baseComment();
			if (this$baseComment == null ? other$baseComment != null : !this$baseComment.equals(other$baseComment)) return false;
			final Object this$commentProducer = this.commentProducer();
			final Object other$commentProducer = other.commentProducer();
			if (this$commentProducer == null ? other$commentProducer != null : !this$commentProducer.equals(other$commentProducer)) return false;
			return true;
		}

		protected boolean canEqual(final Object other) {
			return other instanceof CommentExtensionImpl.CustomEntryData;
		}

		@Override
		public int hashCode() {
			final int PRIME = 59;
			int result = 1;
			final Object $baseComment = this.baseComment();
			result = result * PRIME + ($baseComment == null ? 43 : $baseComment.hashCode());
			final Object $commentProducer = this.commentProducer();
			result = result * PRIME + ($commentProducer == null ? 43 : $commentProducer.hashCode());
			return result;
		}

		@Override
		public String toString() {
			return "CommentExtensionImpl.CustomEntryData(baseComment=" + this.baseComment() + ", commentProducer=" + this.commentProducer() + ")";
		}
	}

	public PatchworkPartAccess<CustomEntryData> customEntryDataAccess() {
		return this.customEntryDataAccess;
	}

	@Nullable
	public PatchworkPartAccess<Boolean> writerInstalledReadWriteContextAccess() {
		return this.writerInstalledReadWriteContextAccess;
	}
}
