package de.siphalor.tweed5.defaultextensions.validation.api.result;

import org.jspecify.annotations.Nullable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.function.Function;

public class ValidationResult<T extends @Nullable Object> {
	private final T value;
	private final Collection<ValidationIssue> issues;
	private final boolean hasError;

	public static <T extends @Nullable Object> ValidationResult<T> ok(T value) {
		return new ValidationResult<>(value, Collections.emptyList(), false);
	}

	public static <T extends @Nullable Object> ValidationResult<T> withIssues(T value, Collection<ValidationIssue> issues) {
		return new ValidationResult<>(value, issues, issuesContainError(issues));
	}

	private static boolean issuesContainError(Collection<ValidationIssue> issues) {
		if (issues.isEmpty()) {
			return false;
		}
		for (ValidationIssue issue : issues) {
			if (issue.level() == ValidationIssueLevel.ERROR) {
				return true;
			}
		}
		return false;
	}

	public ValidationResult<T> andThen(Function<T, ValidationResult<T>> function) {
		if (hasError) {
			return this;
		}
		ValidationResult<T> functionResult = function.apply(value);
		if (functionResult.issues.isEmpty()) {
			if (functionResult.value == value) {
				return this;
			} else if (issues.isEmpty()) {
				return new ValidationResult<>(functionResult.value, Collections.emptyList(), false);
			}
		}
		ArrayList<ValidationIssue> combinedIssues = new ArrayList<>(issues.size() + functionResult.issues.size());
		combinedIssues.addAll(issues);
		combinedIssues.addAll(functionResult.issues);
		return new ValidationResult<>(functionResult.value, combinedIssues, functionResult.hasError);
	}

	public T value() {
		return this.value;
	}

	public Collection<ValidationIssue> issues() {
		return this.issues;
	}

	public boolean hasError() {
		return this.hasError;
	}

	private ValidationResult(final T value, final Collection<ValidationIssue> issues, final boolean hasError) {
		this.value = value;
		this.issues = issues;
		this.hasError = hasError;
	}
}
