package de.siphalor.tweed5.defaultextensions.validation.api.validators;

import de.siphalor.tweed5.core.api.entry.ConfigEntry;
import de.siphalor.tweed5.defaultextensions.validation.api.ConfigEntryValidator;
import de.siphalor.tweed5.defaultextensions.validation.api.result.ValidationIssue;
import de.siphalor.tweed5.defaultextensions.validation.api.result.ValidationIssueLevel;
import de.siphalor.tweed5.defaultextensions.validation.api.result.ValidationResult;
import lombok.*;
import org.jspecify.annotations.Nullable;
import java.util.Collections;

public final class NumberRangeValidator<N extends Number> implements ConfigEntryValidator {
	private final Class<N> numberClass;
	@Nullable
	private final N minimum;
	private final boolean minimumExclusive;
	@Nullable
	private final N maximum;
	private final boolean maximumExclusive;
	private final String description;

	public static <N extends Number> Builder<N> builder(Class<N> numberClass) {
		return new Builder<>(numberClass);
	}

	@Override
	public <T extends @Nullable Object> ValidationResult<T> validate(ConfigEntry<T> configEntry, T value) {
		if (!(value instanceof Number)) {
			return ValidationResult.withIssues(value, Collections.singleton(new ValidationIssue("Value must be numeric, got" + getClassName(value), ValidationIssueLevel.ERROR)));
		}
		if (value.getClass() != numberClass) {
			return ValidationResult.withIssues(value, Collections.singleton(new ValidationIssue("Value is of wrong type, expected " + numberClass.getName() + ", got " + getClassName(value), ValidationIssueLevel.ERROR)));
		}
		Number numberValue = (Number) value;
		if (minimum != null) {
			int minCmp = compare(numberValue, minimum);
			if (minimumExclusive ? minCmp <= 0 : minCmp < 0) {
				//noinspection unchecked
				return ValidationResult.withIssues((T) minimum, Collections.singleton(new ValidationIssue(description + ", got: " + value, ValidationIssueLevel.WARN)));
			}
		}
		if (maximum != null) {
			int maxCmp = compare(numberValue, maximum);
			if (maximumExclusive ? maxCmp >= 0 : maxCmp > 0) {
				//noinspection unchecked
				return ValidationResult.withIssues((T) maximum, Collections.singleton(new ValidationIssue(description + " Got: " + value, ValidationIssueLevel.WARN)));
			}
		}
		return ValidationResult.ok(value);
	}

	private static String getClassName(@Nullable Object value) {
		if (value == null) {
			return "<null>";
		}
		return value.getClass().getName();
	}

	private int compare(Number a, Number b) {
		if (numberClass == Byte.class) {
			return Byte.compare(a.byteValue(), b.byteValue());
		} else if (numberClass == Short.class) {
			return Short.compare(a.shortValue(), b.shortValue());
		} else if (numberClass == Integer.class) {
			return Integer.compare(a.intValue(), b.intValue());
		} else if (numberClass == Long.class) {
			return Long.compare(a.longValue(), b.longValue());
		} else if (numberClass == Float.class) {
			return Float.compare(a.floatValue(), b.floatValue());
		} else {
			return Double.compare(a.doubleValue(), b.doubleValue());
		}
	}

	@Override
	public <T> String description(ConfigEntry<T> configEntry) {
		return description;
	}


	public static class Builder<N extends Number> {
		private final Class<N> numberClass;
		@Nullable
		private N minimum;
		private boolean minimumExclusive;
		@Nullable
		private N maximum;
		private boolean maximumExclusive;

		public Builder<N> greaterThan(N minimum) {
			this.minimumExclusive = true;
			this.minimum = minimum;
			return this;
		}

		public Builder<N> greaterThanOrEqualTo(N minimum) {
			this.minimumExclusive = false;
			this.minimum = minimum;
			return this;
		}

		public Builder<N> lessThan(N maximum) {
			this.maximumExclusive = true;
			this.maximum = maximum;
			return this;
		}

		public Builder<N> lessThanOrEqualTo(N maximum) {
			this.maximumExclusive = false;
			this.maximum = maximum;
			return this;
		}

		public NumberRangeValidator<N> build() {
			return new NumberRangeValidator<>(numberClass, minimum, minimumExclusive, maximum, maximumExclusive, createDescription());
		}

		private String createDescription() {
			if (minimum != null) {
				if (maximum != null) {
					if (minimumExclusive == maximumExclusive) {
						if (minimumExclusive) {
							return "Must be exclusively between " + minimum + " and " + maximum + ".";
						} else {
							return "Must be inclusively between " + minimum + " and " + maximum + ".";
						}
					}
					StringBuilder description = new StringBuilder(40);
					description.append("Must be greater than ");
					if (!minimumExclusive) {
						description.append("or equal to ");
					}
					description.append(minimum).append(" and less than ");
					if (!maximumExclusive) {
						description.append("or equal to ");
					}
					description.append(maximum).append('.');
					return description.toString();
				} else if (minimumExclusive) {
					return "Must be greater than " + minimum + ".";
				} else {
					return "Must be greater than or equal to " + minimum + ".";
				}
			}
			if (maximum != null) {
				if (maximumExclusive) {
					return "Must be less than " + maximum + ".";
				} else {
					return "Must be less than or equal to " + maximum + ".";
				}
			}
			return "";
		}

		private Builder(final Class<N> numberClass) {
			this.numberClass = numberClass;
		}
	}

	public Class<N> numberClass() {
		return this.numberClass;
	}

	@Nullable
	public N minimum() {
		return this.minimum;
	}

	public boolean minimumExclusive() {
		return this.minimumExclusive;
	}

	@Nullable
	public N maximum() {
		return this.maximum;
	}

	public boolean maximumExclusive() {
		return this.maximumExclusive;
	}

	public String description() {
		return this.description;
	}

	@Override
	public boolean equals(final Object o) {
		if (o == this) return true;
		if (!(o instanceof NumberRangeValidator)) return false;
		final NumberRangeValidator<?> other = (NumberRangeValidator<?>) o;
		if (this.minimumExclusive() != other.minimumExclusive()) return false;
		if (this.maximumExclusive() != other.maximumExclusive()) return false;
		final Object this$numberClass = this.numberClass();
		final Object other$numberClass = other.numberClass();
		if (this$numberClass == null ? other$numberClass != null : !this$numberClass.equals(other$numberClass)) return false;
		final Object this$minimum = this.minimum();
		final Object other$minimum = other.minimum();
		if (this$minimum == null ? other$minimum != null : !this$minimum.equals(other$minimum)) return false;
		final Object this$maximum = this.maximum();
		final Object other$maximum = other.maximum();
		if (this$maximum == null ? other$maximum != null : !this$maximum.equals(other$maximum)) return false;
		final Object this$description = this.description();
		final Object other$description = other.description();
		if (this$description == null ? other$description != null : !this$description.equals(other$description)) return false;
		return true;
	}

	@Override
	public int hashCode() {
		final int PRIME = 59;
		int result = 1;
		result = result * PRIME + (this.minimumExclusive() ? 79 : 97);
		result = result * PRIME + (this.maximumExclusive() ? 79 : 97);
		final Object $numberClass = this.numberClass();
		result = result * PRIME + ($numberClass == null ? 43 : $numberClass.hashCode());
		final Object $minimum = this.minimum();
		result = result * PRIME + ($minimum == null ? 43 : $minimum.hashCode());
		final Object $maximum = this.maximum();
		result = result * PRIME + ($maximum == null ? 43 : $maximum.hashCode());
		final Object $description = this.description();
		result = result * PRIME + ($description == null ? 43 : $description.hashCode());
		return result;
	}

	@Override
	public String toString() {
		return "NumberRangeValidator(numberClass=" + this.numberClass() + ", minimum=" + this.minimum() + ", minimumExclusive=" + this.minimumExclusive() + ", maximum=" + this.maximum() + ", maximumExclusive=" + this.maximumExclusive() + ", description=" + this.description() + ")";
	}

	private NumberRangeValidator(final Class<N> numberClass, @Nullable final N minimum, final boolean minimumExclusive, @Nullable final N maximum, final boolean maximumExclusive, final String description) {
		this.numberClass = numberClass;
		this.minimum = minimum;
		this.minimumExclusive = minimumExclusive;
		this.maximum = maximum;
		this.maximumExclusive = maximumExclusive;
		this.description = description;
	}
}
