/*
 * Copyright 2021-2023 Siphalor
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 */

package de.siphalor.wanderingcollector;

import net.fabricmc.fabric.api.util.NbtType;
import net.minecraft.class_1799;
import net.minecraft.class_2487;
import net.minecraft.class_2499;
import net.minecraft.class_2520;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Random;

public class LostItemStorage {
	private static final Random RANDOM = new Random();
	private final List<class_1799> stacks = new ArrayList<>();

	public void read(class_2487 parentNbt) {
		stacks.clear();

		if (parentNbt.method_10545(WanderingCollector.LOST_STACKS_KEY)) {
			if (parentNbt.method_10573(WanderingCollector.LOST_STACKS_KEY, NbtType.LIST)) {
				readStacksFrom(parentNbt, WanderingCollector.LOST_STACKS_KEY);
			} else if (parentNbt.method_10573(WanderingCollector.LOST_STACKS_KEY, NbtType.COMPOUND)) {
				class_2487 ownNbt = parentNbt.method_10562(WanderingCollector.LOST_STACKS_KEY);
				if (ownNbt.method_10573(WanderingCollector.LOST_STACKS_KEY, NbtType.LIST)) {
					readStacksFrom(ownNbt, "Stacks");
				}
			}
		}
	}

	private void readStacksFrom(class_2487 parentNbt, String key) {
		class_2499 listNbt = parentNbt.method_10554(key, NbtType.COMPOUND);
		for (class_2520 stackNbt : listNbt) {
			if (stackNbt instanceof class_2487) {
				stacks.add(class_1799.method_7915((class_2487) stackNbt));
			}
		}
	}

	public void write(class_2487 parentNbt) {
		if (stacks.isEmpty()) {
			return;
		}

		class_2487 ownNbt = new class_2487();
		parentNbt.method_10566(WanderingCollector.LOST_STACKS_KEY, ownNbt);

		class_2499 listNbt = new class_2499();
		for (class_1799 stack : stacks) {
			listNbt.add(stack.method_7953(new class_2487()));
		}
		ownNbt.method_10566("Stacks", listNbt);
	}

	public boolean isEmpty() {
		return stacks.isEmpty();
	}

	public void add(class_1799 newStack) {
		if (WCConfig.combineLostStacks && tryCombine(newStack)) {
			return;
		}

		if (stacks.size() >= WCConfig.maxLostStackAmount) {
			stacks.remove(0);
		}
		stacks.add(newStack);
	}

	private boolean tryCombine(class_1799 newStack) {
		int requiredSpace = newStack.method_7947();
		int space = 0;
		List<class_1799> equalStacks = new ArrayList<>();
		for (class_1799 stack : stacks) {
			if (class_1799.method_31577(stack, newStack)) {
				equalStacks.add(stack);
				space += stack.method_7914() - stack.method_7947();

				if (space >= requiredSpace) {
					break;
				}
			}
		}

		if (space < requiredSpace) {
			return false;
		}

		for (class_1799 equalStack : equalStacks) {
			int move = Math.min(equalStack.method_7914() - equalStack.method_7947(), requiredSpace);
			equalStack.method_7933(move);
			requiredSpace -= move;
			if (requiredSpace <= 0) {
				break;
			}
		}
		return true;
	}

	public Collection<class_1799> poll(int stackCount) {
		stackCount = Math.min(stackCount, stacks.size());
		List<class_1799> result = new ArrayList<>(stackCount);
		switch (WCConfig.offerCreation) {
			case NEWEST: {
				List<class_1799> range = stacks.subList(stacks.size() - stackCount, stacks.size());
				result.addAll(range);
				range.clear();
				break;
			}
			case OLDEST: {
				List<class_1799> range = stacks.subList(0, stackCount);
				result.addAll(range);
				range.clear();
				break;
			}
			case RANDOM:
				for (int i = 0; i < stackCount; i++) {
					int randIndex = RANDOM.nextInt(stacks.size());
					class_1799 stack = stacks.get(randIndex);
					if (stack.method_7947() > stack.method_7914()) {
						result.add(stack.method_7971(stack.method_7914()));
					} else {
						result.add(stack);
						stacks.remove(randIndex);
					}
				}
				break;
		}
		return result;
	}

	public enum PollMode {NEWEST, OLDEST, RANDOM}
}
