/*
 * Copyright 2021-2023 Siphalor
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 */

package de.siphalor.wanderingcollector;

import com.google.common.base.CaseFormat;
import de.siphalor.tweed4.annotated.AConfigBackground;
import de.siphalor.tweed4.annotated.AConfigConstraint;
import de.siphalor.tweed4.annotated.AConfigEntry;
import de.siphalor.tweed4.annotated.ATweedConfig;
import de.siphalor.tweed4.config.ConfigEnvironment;
import de.siphalor.tweed4.config.ConfigScope;
import de.siphalor.tweed4.config.constraints.RangeConstraint;
import de.siphalor.wanderingcollector.util.ConfigRegistryConstraint;
import de.siphalor.wanderingcollector.util.Utils;
import org.apache.logging.log4j.Level;

import java.util.Random;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_2378;
import net.minecraft.class_2960;
import net.minecraft.class_3532;

@ATweedConfig(file = WanderingCollector.MOD_ID, scope = ConfigScope.SMALLEST, environment = ConfigEnvironment.UNIVERSAL, casing = CaseFormat.LOWER_HYPHEN, tailors = "tweed4:coat")
@AConfigBackground("minecraft:textures/block/blue_concrete_powder.png")
public class WCConfig {
	private static final Random random = new Random();

	@AConfigEntry(
			comment = "Include manually dropped items in the Wandering Trader offers"
	)
	public static boolean includeDroppedStacks = false;

	@AConfigEntry(
			constraints = @AConfigConstraint(value = RangeConstraint.class, param = "0.."),
			comment = "The maximum number of stacks that get remembered after loosing them"
	)
	public static int maxLostStackAmount = 64;

	@AConfigEntry(
			comment = "Defines in which order lost stacks will be used for the Wandering Traders' offers." +
					" - NEWEST: The newest lost stacks will be used first." +
					" - OLDEST: The oldest lost stacks will be used first." +
					" - RANDOM: Lost stacks will be used in a random order."
	)
	public static LostItemStorage.PollMode offerCreation = LostItemStorage.PollMode.RANDOM;

	@AConfigEntry(
			comment = "Lost stacks that are stackable will be tried to be merged into existing stacks"
	)
	public static boolean combineLostStacks = true;

	@AConfigEntry(
			constraints = @AConfigConstraint(value = RangeConstraint.class, param = "0.."),
			comment = "The number of buy-back trades Wandering Traders will offer"
	)
	public static int buyBackTrades = 2;

	public static PriceDefinition defaultPrices;

	@AConfigBackground("minecraft:textures/block/green_terracotta.png")
	public static class PriceDefinition {
		@AConfigEntry(
				constraints = @AConfigConstraint(value = RangeConstraint.class, param = "1..64"),
				comment = "The minimum emerald price to buy back a stack"
		)
		public int minPrice = 32;

		@AConfigEntry(
				constraints = @AConfigConstraint(value = RangeConstraint.class, param = "1..64"),
				comment = "The maximum emerald price to buy back a stack"
		)
		public int maxPrice = 64;

		@AConfigEntry(
				constraints = @AConfigConstraint(value = ConfigRegistryConstraint.class, param = "minecraft:item"),
				comment = "The item that must be paid with for the trade offer"
		)
		public String priceItem = "minecraft:emerald";

		@AConfigEntry(
				comment = "Whether the price will be scaled with the amount of items in the stack."
		)
		public boolean scalePriceWithCount = true;

		@AConfigEntry(
				constraints = @AConfigConstraint(value = RangeConstraint.class, param = "0..1"),
				comment = "The minimum percentage that scale-price-with-amount will scale to."
		)
		public float minPriceScale = 0.3f;

		public class_1799 getPriceStack(class_1799 forStack) {
			class_1792 priceItem = class_2378.field_11142.method_10223(new class_2960(this.priceItem));
			if (priceItem == class_1802.field_8162) {
				WanderingCollector.log(Level.WARN, "Unknown price item \"" + this.priceItem + "\", defaulting to minecraft:emerald");
				priceItem = class_1802.field_8687;
			}
			class_1799 result = new class_1799(priceItem);

			int count = Utils.randInclusive(random, this.minPrice, this.maxPrice);
			if (scalePriceWithCount) {
				count = class_3532.method_15386(count * (minPriceScale + (1 - minPriceScale) * forStack.method_7947() / forStack.method_7914()));
			}
			result.method_7939(count);
			return result;
		}
	}
}
