package io.github.vampirestudios.vampirelib;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Consumer;

import javax.annotation.Nullable;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_2960;
import net.minecraft.class_4722;
import net.minecraft.class_4730;
import io.github.vampirestudios.vampirelib.callbacks.client.TextureStitchCallback;

/**
 * Manager class for texture information for Blueprint Chests.
 *
 * @author SmellyModder (Luke Tonon)
 */
public final class ChestManager {
	public static final Map<String, ChestInfo> CHEST_INFO_MAP = new HashMap<>();

	/**
	 * Puts a created {@link ChestInfo} onto the {@link #CHEST_INFO_MAP} for a given ID and type.
	 *
	 * @param modId   Mod ID for the chest.
	 * @param type    Type for the chest. (e.g. "oak")
	 * @param trapped If the chest is trapped.
	 */
	public static synchronized void putChestInfo(String modId, String type, boolean trapped) {
		CHEST_INFO_MAP.put(modId + ":" + type + (trapped ? "_trapped" : ""), new ChestInfo(modId, type, trapped));
	}

	/**
	 * Gets the {@link ChestInfo} for a given chest type.
	 *
	 * @param chestType A string for the {@link ChestInfo} to lookup.
	 * @return The {@link ChestInfo} for the given chest type, or null if there is no {@link ChestInfo} for the given type.
	 */
	@Nullable
	public static ChestInfo getInfoForChest(String chestType) {
		return CHEST_INFO_MAP.get(chestType);
	}

	public static class ChestInfo {
		private final class_2960 single, left, right;
		@Environment(EnvType.CLIENT)
		private class_4730 singleMaterial, leftMaterial, rightMaterial;

		public ChestInfo(String modId, String type, boolean trapped) {
			String chest = trapped ? "trapped" : "normal";
			this.single = new class_2960(modId, "entity/chest/" + type + "/" + chest);
			this.left = new class_2960(modId, "entity/chest/" + type + "/" + chest + "_left");
			this.right = new class_2960(modId, "entity/chest/" + type + "/" + chest + "_right");
		}

		/**
		 * Adds the internal textures to the stitch event and initializes the {@link class_4730}s.
		 *
		 * @param event A {@link TextureStitchCallback.Pre} to setup this info from.
		 */
		@Environment(EnvType.CLIENT)
		public void setup(Consumer<class_2960> event) {
			event.accept(this.single);
			event.accept(this.left);
			event.accept(this.right);
			this.singleMaterial = new class_4730(class_4722.field_21709, this.single);
			this.leftMaterial = new class_4730(class_4722.field_21709, this.left);
			this.rightMaterial = new class_4730(class_4722.field_21709, this.right);
		}

		/**
		 * Gets this info's default/single {@link class_4730}.
		 *
		 * @return This info's default/single {@link class_4730}.
		 */
		@Environment(EnvType.CLIENT)
		public class_4730 getSingleMaterial() {
			return this.singleMaterial;
		}

		/**
		 * Gets this info's left {@link class_4730}.
		 *
		 * @return This info's left {@link class_4730}.
		 */
		@Environment(EnvType.CLIENT)
		public class_4730 getLeftMaterial() {
			return this.leftMaterial;
		}

		/**
		 * Gets this info's right {@link class_4730}.
		 *
		 * @return This info's right {@link class_4730}.
		 */
		@Environment(EnvType.CLIENT)
		public class_4730 getRightMaterial() {
			return this.rightMaterial;
		}
	}
}