/*
 * Minecraft Forge
 * Copyright (c) 2016-2021.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation version 2.1
 * of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

package io.github.vampirestudios.vampirelib.api;

import java.util.ArrayDeque;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import net.minecraft.class_1959;
import net.minecraft.class_1972;
import net.minecraft.class_2378;
import net.minecraft.class_2960;
import net.minecraft.class_5321;
import net.minecraft.class_5458;
import com.google.common.collect.ImmutableList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BiomeDictionary {
	private static final boolean DEBUG = false;
	private static final Logger LOGGER = LoggerFactory.getLogger(BiomeDictionary.class);

	public static final class Type {
		private static final Map<String, Type> byName = new TreeMap<>();
		private static final Collection<Type> allTypes = Collections.unmodifiableCollection(byName.values());

		/*Temperature-based tags. Specifying neither implies a biome is temperate*/
		public static final Type HOT = new Type("HOT");
		public static final Type COLD = new Type("COLD");

		//Tags specifying the amount of vegetation a biome has. Specifying neither implies a biome to have moderate amounts*/
		public static final Type SPARSE = new Type("SPARSE");
		public static final Type DENSE = new Type("DENSE");

		//Tags specifying how moist a biome is. Specifying neither implies the biome as having moderate humidity*/
		public static final Type WET = new Type("WET");
		public static final Type DRY = new Type("DRY");

		/*Tree-based tags, SAVANNA refers to dry, desert-like trees (Such as Acacia), CONIFEROUS refers to snowy trees (Such as Spruce) and JUNGLE refers to jungle trees.
		 * Specifying no tag implies a biome has temperate trees (Such as Oak)*/
		public static final Type SAVANNA = new Type("SAVANNA");
		public static final Type CONIFEROUS = new Type("CONIFEROUS");
		public static final Type JUNGLE = new Type("JUNGLE");

		/*Tags specifying the nature of a biome*/
		public static final Type SPOOKY = new Type("SPOOKY");
		public static final Type DEAD = new Type("DEAD");
		public static final Type LUSH = new Type("LUSH");
		public static final Type MUSHROOM = new Type("MUSHROOM");
		public static final Type MAGICAL = new Type("MAGICAL");
		public static final Type RARE = new Type("RARE");
		public static final Type PLATEAU = new Type("PLATEAU");
		public static final Type MODIFIED = new Type("MODIFIED");

		public static final Type OCEAN = new Type("OCEAN");
		public static final Type RIVER = new Type("RIVER");
		/**
		 * A general tag for all water-based biomes. Shown as present if OCEAN or RIVER are.
		 **/
		public static final Type WATER = new Type("WATER", OCEAN, RIVER);

		/*Generic types which a biome can be*/
		public static final Type MESA = new Type("MESA");
		public static final Type FOREST = new Type("FOREST");
		public static final Type PLAINS = new Type("PLAINS");
		public static final Type HILLS = new Type("HILLS");
		public static final Type SWAMP = new Type("SWAMP");
		public static final Type SANDY = new Type("SANDY");
		public static final Type SNOWY = new Type("SNOWY");
		public static final Type WASTELAND = new Type("WASTELAND");
		public static final Type BEACH = new Type("BEACH");
		public static final Type VOID = new Type("VOID");
		public static final Type UNDERGROUND = new Type("UNDERGROUND");

		/*Mountain related tags*/
		public static final Type PEAK = new Type("PEAK");
		public static final Type SLOPE = new Type("SLOPE");
		public static final Type MOUNTAIN = new Type("MOUNTAIN", PEAK, SLOPE);

		/*Tags specifying the dimension a biome generates in. Specifying none implies a biome that generates in a modded dimension*/
		public static final Type OVERWORLD = new Type("OVERWORLD");
		public static final Type NETHER = new Type("NETHER");
		public static final Type END = new Type("END");

		private final String name;
		private final List<Type> subTypes;
		private final Set<class_5321<class_1959>> biomes = new HashSet<>();
		private final Set<class_5321<class_1959>> biomesUn = Collections.unmodifiableSet(biomes);

		private Type(String name, Type... subTypes) {
			this.name = name;
			this.subTypes = ImmutableList.copyOf(subTypes);

			byName.put(name, this);
		}

		/**
		 * Gets the name for this type.
		 */
		public String getName() {
			return name;
		}

		public String toString() {
			return name;
		}

		/**
		 * Retrieves a Type instance by name,
		 * if one does not exist already it creates one.
		 * This can be used as intermediate measure for modders to
		 * add their own Biome types.
		 * <p>
		 * There are <i>no</i> naming conventions besides:
		 * <ul><li><b>Must</b> be all upper case (enforced by name.toUpper())</li>
		 * <li><b>No</b> Special characters. {Unenforced, just don't be a pain, if it becomes a issue I WILL
		 * make this RTE with no worry about backwards compatibility}</li></ul>
		 * <p>
		 * Note: For performance sake, the return value of this function SHOULD be cached.
		 * Two calls with the same name SHOULD return the same value.
		 *
		 * @param name The name of this Type
		 * @return An instance of Type for this name.
		 */
		public static Type getType(String name, Type... subTypes) {
			name = name.toUpperCase();
			Type t = byName.get(name);
			if (t == null) {
				t = new Type(name, subTypes);
			}
			return t;
		}

		/**
		 * Checks if a type instance exists for a given name. Does not have any side effects if a type does not already exist.
		 * This can be used for checking if a user-defined type is valid, for example, in a codec which accepts biome dictionary names.
		 *
		 * @param name The name.
		 * @return {@code true} if a type exists with this name.
		 * @see #getType(String, Type...) #getType for type naming conventions.
		 */
		public static boolean hasType(String name) {
			return byName.containsKey(name.toUpperCase());
		}

		/**
		 * @return An unmodifiable collection of all current biome types.
		 */
		public static Collection<Type> getAll() {
			return allTypes;
		}
	}

	private static final Map<class_5321<class_1959>, BiomeInfo> biomeInfoMap = new HashMap<>();

	private static class BiomeInfo {
		private final Set<Type> types = new HashSet<>();
		private final Set<Type> typesUn = Collections.unmodifiableSet(this.types);
	}

	public static void init() {}

	static {
		registerVanillaBiomes();
	}

	/**
	 * Adds the given types to the biome.
	 */
	public static void addTypes(class_5321<class_1959> biome, Type... types) {
		Collection<Type> supertypes = listSupertypes(types);
		Collections.addAll(supertypes, types);

		for (Type type : supertypes) {
			type.biomes.add(biome);
		}

		BiomeInfo biomeInfo = getBiomeInfo(biome);
		Collections.addAll(biomeInfo.types, types);
		biomeInfo.types.addAll(supertypes);
	}

	/**
	 * Gets the set of biomes that have the given type.
	 */
	public static Set<class_5321<class_1959>> getBiomes(Type type) {
		return type.biomesUn;
	}

	/**
	 * Gets the set of types that have been added to the given biome.
	 */
	public static Set<Type> getTypes(class_5321<class_1959> biome) {
		return getBiomeInfo(biome).typesUn;
	}

	/**
	 * Checks if the two given biomes have types in common.
	 *
	 * @return returns true if a common type is found, false otherwise
	 */
	public static boolean areSimilar(class_5321<class_1959> biomeA, class_5321<class_1959> biomeB) {
		Set<Type> typesA = getTypes(biomeA);
		Set<Type> typesB = getTypes(biomeB);
		return typesA.stream().anyMatch(typesB::contains);
	}

	/**
	 * Checks if the given type has been added to the given biome.
	 */
	public static boolean hasType(class_5321<class_1959> biome, Type type) {
		return getTypes(biome).contains(type);
	}

	/**
	 * Checks if any type has been added to the given biome.
	 */
	public static boolean hasAnyType(class_5321<class_1959> biome) {
		return !getBiomeInfo(biome).types.isEmpty();
	}

	//Internal implementation
	private static BiomeInfo getBiomeInfo(class_5321<class_1959> biome) {
		return biomeInfoMap.computeIfAbsent(biome, k -> new BiomeInfo());
	}

	private static Collection<Type> listSupertypes(Type... types) {
		Set<Type> supertypes = new HashSet<Type>();
		Deque<Type> next = new ArrayDeque<Type>();
		Collections.addAll(next, types);

		while (!next.isEmpty()) {
			Type type = next.remove();

			for (Type sType : Type.byName.values()) {
				if (sType.subTypes.contains(type) && supertypes.add(sType))
					next.add(sType);
			}
		}

		return supertypes;
	}

	private static void registerVanillaBiomes() {
		addTypes(class_1972.field_9423, Type.OCEAN, Type.OVERWORLD);
		addTypes(class_1972.field_9451, Type.PLAINS, Type.OVERWORLD);
		addTypes(class_1972.field_9424, Type.HOT, Type.DRY, Type.SANDY, Type.OVERWORLD);
		addTypes(class_1972.field_35116, Type.HILLS, Type.OVERWORLD);
		addTypes(class_1972.field_9409, Type.FOREST, Type.OVERWORLD);
		addTypes(class_1972.field_9420, Type.COLD, Type.CONIFEROUS, Type.FOREST, Type.OVERWORLD);
		addTypes(class_1972.field_9471, Type.WET, Type.SWAMP, Type.OVERWORLD);
		addTypes(class_1972.field_9438, Type.RIVER, Type.OVERWORLD);
		addTypes(class_1972.field_9461, Type.HOT, Type.DRY, Type.NETHER);
		addTypes(class_1972.field_9411, Type.COLD, Type.DRY, Type.END);
		addTypes(class_1972.field_9435, Type.COLD, Type.OCEAN, Type.SNOWY, Type.OVERWORLD);
		addTypes(class_1972.field_9463, Type.COLD, Type.RIVER, Type.SNOWY, Type.OVERWORLD);
		addTypes(class_1972.field_35117, Type.COLD, Type.SNOWY, Type.WASTELAND, Type.OVERWORLD);
		addTypes(class_1972.field_9462, Type.MUSHROOM, Type.RARE, Type.OVERWORLD);
		addTypes(class_1972.field_9434, Type.BEACH, Type.OVERWORLD);
		addTypes(class_1972.field_9417, Type.HOT, Type.WET, Type.DENSE, Type.JUNGLE, Type.OVERWORLD);
		addTypes(class_1972.field_35118, Type.HOT, Type.WET, Type.JUNGLE, Type.FOREST, Type.RARE, Type.OVERWORLD);
		addTypes(class_1972.field_9446, Type.OCEAN, Type.OVERWORLD);
		addTypes(class_1972.field_9419, Type.BEACH, Type.OVERWORLD);
		addTypes(class_1972.field_9478, Type.COLD, Type.BEACH, Type.SNOWY, Type.OVERWORLD);
		addTypes(class_1972.field_9412, Type.FOREST, Type.OVERWORLD);
		addTypes(class_1972.field_9475, Type.SPOOKY, Type.DENSE, Type.FOREST, Type.OVERWORLD);
		addTypes(class_1972.field_9454, Type.COLD, Type.CONIFEROUS, Type.FOREST, Type.SNOWY, Type.OVERWORLD);
		addTypes(class_1972.field_35119, Type.COLD, Type.CONIFEROUS, Type.FOREST, Type.OVERWORLD);
		addTypes(class_1972.field_35120, Type.HILLS, Type.FOREST, Type.SPARSE, Type.OVERWORLD);
		addTypes(class_1972.field_9449, Type.HOT, Type.SAVANNA, Type.PLAINS, Type.SPARSE, Type.OVERWORLD);
		addTypes(class_1972.field_9430, Type.HOT, Type.SAVANNA, Type.PLAINS, Type.SPARSE, Type.RARE, Type.OVERWORLD, Type.SLOPE, Type.PLATEAU);
		addTypes(class_1972.field_9415, Type.MESA, Type.SANDY, Type.DRY, Type.OVERWORLD);
		addTypes(class_1972.field_35110, Type.MESA, Type.SANDY, Type.DRY, Type.SPARSE, Type.OVERWORLD, Type.SLOPE, Type.PLATEAU);
		addTypes(class_1972.field_34470, Type.PLAINS, Type.PLATEAU, Type.SLOPE, Type.OVERWORLD);
		addTypes(class_1972.field_34471, Type.COLD, Type.CONIFEROUS, Type.FOREST, Type.SNOWY, Type.SLOPE, Type.OVERWORLD);
		addTypes(class_1972.field_34472, Type.COLD, Type.SPARSE, Type.SNOWY, Type.SLOPE, Type.OVERWORLD);
		addTypes(class_1972.field_34474, Type.COLD, Type.SPARSE, Type.SNOWY, Type.PEAK, Type.OVERWORLD);
		addTypes(class_1972.field_35115, Type.COLD, Type.SPARSE, Type.SNOWY, Type.PEAK, Type.OVERWORLD);
		addTypes(class_1972.field_34475, Type.HOT, Type.PEAK, Type.OVERWORLD);
		addTypes(class_1972.field_9457, Type.END);
		addTypes(class_1972.field_9447, Type.END);
		addTypes(class_1972.field_9442, Type.END);
		addTypes(class_1972.field_9465, Type.END);
		addTypes(class_1972.field_9408, Type.OCEAN, Type.HOT, Type.OVERWORLD);
		addTypes(class_1972.field_9441, Type.OCEAN, Type.OVERWORLD);
		addTypes(class_1972.field_9467, Type.OCEAN, Type.COLD, Type.OVERWORLD);
		addTypes(class_1972.field_9439, Type.OCEAN, Type.OVERWORLD);
		addTypes(class_1972.field_9470, Type.OCEAN, Type.COLD, Type.OVERWORLD);
		addTypes(class_1972.field_9418, Type.OCEAN, Type.COLD, Type.OVERWORLD);
		addTypes(class_1972.field_9473, Type.VOID);
		addTypes(class_1972.field_9455, Type.PLAINS, Type.RARE, Type.OVERWORLD);
		addTypes(class_1972.field_35111, Type.HILLS, Type.SPARSE, Type.RARE, Type.OVERWORLD);
		addTypes(class_1972.field_9414, Type.FOREST, Type.RARE, Type.OVERWORLD);
		addTypes(class_1972.field_9453, Type.COLD, Type.SNOWY, Type.RARE, Type.OVERWORLD);
		addTypes(class_1972.field_35112, Type.FOREST, Type.DENSE, Type.RARE, Type.OVERWORLD);
		addTypes(class_1972.field_35113, Type.DENSE, Type.FOREST, Type.RARE, Type.OVERWORLD);
		addTypes(class_1972.field_35114, Type.HOT, Type.DRY, Type.SPARSE, Type.SAVANNA, Type.HILLS, Type.RARE, Type.OVERWORLD);
		addTypes(class_1972.field_9443, Type.MESA, Type.HOT, Type.DRY, Type.SPARSE, Type.RARE, Type.OVERWORLD);
		addTypes(class_1972.field_9440, Type.HOT, Type.WET, Type.RARE, Type.JUNGLE, Type.OVERWORLD);
		addTypes(class_1972.field_29218, Type.UNDERGROUND, Type.LUSH, Type.WET, Type.OVERWORLD);
		addTypes(class_1972.field_28107, Type.UNDERGROUND, Type.SPARSE, Type.OVERWORLD);
		addTypes(class_1972.field_22076, Type.HOT, Type.DRY, Type.NETHER);
		addTypes(class_1972.field_22077, Type.HOT, Type.DRY, Type.NETHER, Type.FOREST);
		addTypes(class_1972.field_22075, Type.HOT, Type.DRY, Type.NETHER, Type.FOREST);
		addTypes(class_1972.field_23859, Type.HOT, Type.DRY, Type.NETHER);

		if (DEBUG) {
			StringBuilder buf = new StringBuilder();
			buf.append("BiomeDictionary:\n");
			Type.byName.forEach((name, type) ->
				buf.append("    ").append(type.name).append(": ")
					.append(type.biomes.stream()
						.map(class_5321::method_29177)
						.sorted(class_2960::method_12833)
						.map(Object::toString)
						.collect(Collectors.joining(", "))
					)
					.append('\n')
			);

			boolean missing = false;
			List<class_5321<class_1959>> all = StreamSupport.stream(class_5458.field_25933.spliterator(), false)
				.map(b -> class_5321.method_29179(class_2378.field_25114, class_5458.field_25933.method_10221(b)))
				.sorted().toList();

			for (class_5321<class_1959> key : all) {
				if (!biomeInfoMap.containsKey(key)) {
					if (!missing) {
						buf.append("Missing:\n");
						missing = true;
					}
					buf.append("    ").append(key.method_29177()).append('\n');
				}
			}
			LOGGER.debug(buf.toString());
		}
	}
}
