/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package io.github.vampirestudios.vampirelib.api;

import java.util.Objects;
import java.util.function.Function;

import com.google.common.base.Preconditions;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.api.datagen.v1.FabricDataGenerator;
import net.fabricmc.fabric.api.datagen.v1.provider.FabricTagProvider;
import net.fabricmc.fabric.impl.datagen.FabricDataGenHelper;
import net.fabricmc.fabric.mixin.datagen.DynamicRegistryManagerAccessor;
import net.minecraft.class_1291;
import net.minecraft.class_1299;
import net.minecraft.class_1792;
import net.minecraft.class_2248;
import net.minecraft.class_2378;
import net.minecraft.class_2474;
import net.minecraft.class_2960;
import net.minecraft.class_3494;
import net.minecraft.class_5216;
import net.minecraft.class_5284;
import net.minecraft.class_5321;
import net.minecraft.class_6862;

public abstract class CustomTagProviders<T> extends FabricTagProvider<T> {

	/**
	 * Construct a new {@link FabricTagProvider}.
	 *
	 * <p>Common implementations of this class are provided. For example @see BlockTagProvider
	 *
	 * @param dataGenerator The data generator instance
	 * @param registry The backing registry for the Tag type.
	 * @param path The directory name to write the tag file names. Example: "blocks" or "items"
	 */

	protected CustomTagProviders(FabricDataGenerator dataGenerator, class_2378<T> registry, String path) {
		super(dataGenerator, registry, path);
	}

	public CustomTagProviders<T>.CustomFabricTagBuilder<T> getOrCreateTagBuilderCustom(class_6862<T> tag) {
		return new CustomTagProviders.CustomFabricTagBuilder(super.method_10512(tag));
	}

	@Override
	public CustomFabricTagBuilder<T> method_10512(class_6862<T> tag) {
		return new CustomFabricTagBuilder<>(super.method_10512(tag));
	}

	public abstract static class CustomBlockTagProvider extends CustomTagProviders<class_2248> {
		protected CustomBlockTagProvider(FabricDataGenerator dataGenerator) {
			super(dataGenerator, class_2378.field_11146, "blocks");
		}

		public CustomFabricTagBuilder<class_2248> tagCustom(class_6862<class_2248> tag) {
			return getOrCreateTagBuilderCustom(tag);
		}
	}

	public abstract static class CustomItemTagProvider extends CustomTagProviders<class_1792> {
		@Nullable
		private final Function<class_6862<class_2248>, class_3494.class_3495> blockTagBuilderProvider;

		protected CustomItemTagProvider(FabricDataGenerator dataGenerator, @Nullable CustomBlockTagProvider blockTagProvider) {
			super(dataGenerator, class_2378.field_11142, "items");

			this.blockTagBuilderProvider = blockTagProvider == null ? null : blockTagProvider::method_27169;
		}

		/**
		 * Construct an {@link ItemTagProvider} tag provider <b>without</b> an associated {@link BlockTagProvider} tag provider.
		 *
		 * @param dataGenerator a {@link ItemTagProvider} tag provider
		 */
		public CustomItemTagProvider(FabricDataGenerator dataGenerator) {
			this(dataGenerator, null);
		}

		/**
		 * Copy the entries from a tag with the {@link class_2248} type into this item tag.
		 *
		 * <p>The {@link ItemTagProvider} tag provider must be constructed with an associated {@link BlockTagProvider} tag provider to use this method.
		 *
		 * <p>Any block ids that do not exist in the item registry will be filtered out automatically.
		 *
		 * @param blockTag The block tag to copy from.
		 * @param itemTag The item tag to copy to.
		 */
		public void copy(class_6862<class_2248> blockTag, class_6862<class_1792> itemTag) {
			class_3494.class_3495 blockTagBuilder = Objects.requireNonNull(this.blockTagBuilderProvider, "Pass Block tag provider via constructor to use copy").apply(blockTag);
			class_3494.class_3495 itemTagBuilder = this.method_27169(itemTag);
			blockTagBuilder.method_26785().filter((entry) -> entry.comp_324().method_32832(this.field_11482::method_10250, (id) -> true)).forEach(itemTagBuilder::method_27064);
		}

		public CustomFabricTagBuilder<class_1792> tagCustom(class_6862<class_1792> tag) {
			return getOrCreateTagBuilderCustom(tag);
		}
	}

	public abstract static class VEntityTagProvider extends CustomTagProviders<class_1299<?>> {
		public VEntityTagProvider(FabricDataGenerator dataGenerator) {
			super(dataGenerator, class_2378.field_11145, "entity_types");
		}

		public CustomFabricTagBuilder<class_1299<?>> tagCustom(class_6862<class_1299<?>> tag) {
			return new CustomFabricTagBuilder<class_1299<?>>(super.method_10512(tag));
		}
	}

	/**
	 * Extend this class to create dynamic registry tags.
	 */
	public abstract static class DynamicRegistryTagProvider<T> extends CustomTagProviders<T> {
		/**
		 * Construct a new {@link CustomTagProviders.DynamicRegistryTagProvider}.
		 *
		 * @param dataGenerator The data generator instance
		 * @param registryKey The registry key of the dynamic registry
		 * @param path The directory name to write the tag file names
		 * @throws IllegalArgumentException if the registry is static registry
		 */
		protected DynamicRegistryTagProvider(FabricDataGenerator dataGenerator, class_5321<? extends class_2378<T>> registryKey, String path) {
			super(dataGenerator, FabricDataGenHelper.getFakeDynamicRegistry(), path);
			Preconditions.checkArgument(DynamicRegistryManagerAccessor.getInfos().containsKey(registryKey), "Only dynamic registries are supported in this tag provider.");
		}
	}

	/**
	 * Extend this class to create {@link class_1291} tags in the "/mob_effects" tag directory.
	 */

	public abstract static class MobEffectTagProvider extends CustomTagProviders<class_1291> {
		protected MobEffectTagProvider(FabricDataGenerator dataGenerator) {
			super(dataGenerator, class_2378.field_11159, "mob_effects");
		}

		public CustomFabricTagBuilder<class_1291> tagCustom(class_6862<class_1291> tag) {
			return new CustomFabricTagBuilder<class_1291>(super.method_10512(tag));
		}
	}

	public abstract static class NoiseSettingsTagProvider extends io.github.vampirestudios.vampirelib.api.CustomTagProviders.DynamicRegistryTagProvider<class_5284> {
		protected NoiseSettingsTagProvider(FabricDataGenerator dataGenerator) {
			super(dataGenerator, class_2378.field_26374, "worldgen/noise_settings");
		}

		public CustomFabricTagBuilder<class_5284> tagCustom(class_6862<class_5284> tag) {
			return new CustomFabricTagBuilder<class_5284>(super.method_10512(tag));
		}
	}

	/*public abstract static class DimensionTypeTagProvider extends CustomTagProviders<DimensionType> {
		protected DimensionTypeTagProvider(FabricDataGenerator dataGenerator) {
			super(dataGenerator, Minecraft.getInstance().getSingleplayerServer().overworld().registryAccess()
				.registryOrThrow(Registry.DIMENSION_TYPE_REGISTRY), "worldgen/dimension_type", "Dimension Type Tags");
		}

		public CustomFabricTagBuilder<DimensionType> tagCustom(TagKey<DimensionType> tag) {
			return new CustomFabricTagBuilder<DimensionType>(super.tag(tag));
		}
	}*/

	/*public abstract static class DimensionTagProvider extends CustomTagProviders<Level> {
		protected DimensionTagProvider(FabricDataGenerator dataGenerator) {
			super(dataGenerator, BuiltinRegistries.DI, "worldgen/dimension", "Dimension Tags");
		}

		public CustomFabricTagBuilder<Level> tagCustom(TagKey<Level> tag) {
			return new CustomFabricTagBuilder<Level>(super.tag(tag));
		}
	}*/

	/**
	 * Extend this class to create {@link class_5216.class_5487} tags in the "worldgen/biomes" tag directory.
	 */

	public abstract static class NoiseTagProvider extends io.github.vampirestudios.vampirelib.api.CustomTagProviders.DynamicRegistryTagProvider<class_5216.class_5487> {
		protected NoiseTagProvider(FabricDataGenerator dataGenerator) {
			super(dataGenerator, class_2378.field_35433, "worldgen/noises");
		}

		public CustomFabricTagBuilder<class_5216.class_5487> tagCustom(class_6862<class_5216.class_5487> tag) {
			return new CustomFabricTagBuilder<class_5216.class_5487>(super.method_10512(tag));
		}
	}

	public final class CustomFabricTagBuilder<T> extends class_2474.class_5124<T> {
		private final class_2474.class_5124<T> parent;

		public CustomFabricTagBuilder(class_2474.class_5124<T> parent) {
			super(parent.field_23960, parent.field_23961, parent.field_23962);
			this.parent = parent;
		}

		/**
		 * Set the value of the `replace` flag in a Tag.
		 *
		 * <p>When set to true the tag will replace any existing tag entries.
		 *
		 * @return the {@link CustomFabricTagBuilder} instance
		 */
		public CustomFabricTagBuilder<T> setReplace(boolean replace) {
			((net.fabricmc.fabric.impl.datagen.FabricTagBuilder) field_23960).fabric_setReplace(replace);
			return this;
		}

		/**
		 * Add a single element to the tag.
		 *
		 * @return the {@link CustomFabricTagBuilder} instance
		 */
		@Override
		public CustomFabricTagBuilder<T> method_26793(T element) {
			assertStaticRegistry();
			parent.method_26793(element);
			return this;
		}

		/**
		 * Add a single element to the tag.
		 *
		 * @return the {@link CustomFabricTagBuilder} instance
		 */
		public CustomFabricTagBuilder<T> add(class_2960 id) {
			field_23960.method_26784(id, field_23962);
			return this;
		}

		/**
		 * Add a single element to the tag.
		 *
		 * @return the {@link CustomFabricTagBuilder} instance
		 */
		public CustomFabricTagBuilder<T> add(class_5321<? extends T> registryKey) {
			return add(registryKey.method_29177());
		}

		/**
		 * Add an optional {@link class_2960} to the tag.
		 *
		 * @return the {@link CustomFabricTagBuilder} instance
		 */
		@Override
		public CustomFabricTagBuilder<T> method_35922(@NotNull class_2960 id) {
			parent.method_35922(id);
			return this;
		}

		/**
		 * Add another tag to this tag.
		 *
		 * @return the {@link CustomFabricTagBuilder} instance
		 */
		@Override
		public CustomFabricTagBuilder<T> method_26792(@NotNull class_6862<T> tag) {
			parent.method_26792(tag);
			return this;
		}

		/**
		 * Add another optional tag to this tag.
		 *
		 * @return the {@link CustomFabricTagBuilder} instance
		 */
		@Override
		public CustomFabricTagBuilder<T> method_35923(class_2960 id) {
			parent.method_35923(id);
			return this;
		}

		/**
		 * Add another tag to this tag, ignoring any warning.
		 *
		 * <p><b>Note:</b> only use this method if you sure that the tag will be always available at runtime.
		 * If not, use {@link #method_35923(class_2960)} instead.
		 *
		 * @return the {@link CustomFabricTagBuilder} instance
		 */
		public CustomFabricTagBuilder<T> forceAddTag(class_6862<T> tag) {
			field_23960.method_26784(tag.comp_327(), field_23962);
			return this;
		}

		/**
		 * Add multiple elements to this tag.
		 *
		 * @return the {@link CustomFabricTagBuilder} instance
		 */
		@SafeVarargs
		@Override
		public final CustomFabricTagBuilder<T> method_26795(T... elements) {
			assertStaticRegistry();

			for (T element : elements) {
				method_26793(element);
			}
			return this;
		}

		/**
		 * Add multiple elements to this tag.
		 *
		 * @return the {@link CustomFabricTagBuilder} instance
		 */
		public CustomFabricTagBuilder<T> add(class_2960... ids) {
			for (class_2960 id : ids) {
				add(id);
			}
			return this;
		}

		/**
		 * Add another tag to this tag.
		 *
		 * @return the {@link FabricTagProvider.FabricTagBuilder} instance
		 */
		@SafeVarargs
		public final CustomFabricTagBuilder<T> addTags(class_6862<T>... values) {
			for (class_6862<T> value : values) {
				this.method_26792(value);
			}
			return this;
		}

		private void assertStaticRegistry() {
			if (CustomTagProviders.this instanceof io.github.vampirestudios.vampirelib.api.CustomTagProviders.DynamicRegistryTagProvider) {
				throw new UnsupportedOperationException("Adding object instances is not supported for DynamicRegistryTagProvider.");
			}
		}

	}

}
