package io.github.vampirestudios.vampirelib.api.blockspreading;

import com.google.common.collect.HashBasedTable;
import com.google.common.collect.Table;
import org.slf4j.Logger;

import com.mojang.logging.LogUtils;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2680;

public class SpreadBehaviors {

	private static final Logger LOGGER = LogUtils.getLogger();
	private static final Table<class_2248, BlockSpreadingType, SpreadingBehavior> SPREADERS = HashBasedTable.create();

	static {
		setupVanillaBehavior();
	}

	/**
	 * This method allows you to add spreading behavior, which is used by {@link SpreadingBlock}
	 */
	private static void addSpreaderBehavior(class_2248 block, BlockSpreadingType type, SpreadingBehavior behavior) {
		if (SPREADERS.contains(block, type))
			LOGGER.info("Replacing spreading behavior for block '{}' and spreader type '{}'", block, type.getName());
		SPREADERS.put(block, type, behavior);
	}

	/**
	 * This method allows you to add a simple spreading behavior
	 */
	public static void addSimpleSpreaderBehavior(class_2248 block, BlockSpreadingType type) {
		addSpreaderBehavior(block, type, new SimpleSpreaderBehavior(block.method_9564()));
	}

	/**
	 * This method allows you to add a complex spreading behavior
	 */
	public static void addComplexSpreaderBehavior(class_2248 block, BlockSpreadingType type, SpreadingBehavior behavior) {
		addSpreaderBehavior(block, type, behavior);
	}

	public static boolean canSpread(class_2680 state, BlockSpreadingType type) {
		return getSpreadingBehavior(state.method_26204(), type) != null;
	}

	public static class_2680 getSpreadState(class_2680 state, class_1937 level, class_2338 pos, BlockSpreadingType type) {
		SpreadingBehavior behavior = getSpreadingBehavior(state.method_26204(), type);
		if (behavior == null) return state;
		return behavior.getStateForSpread(state, level, pos);
	}

	private static SpreadingBehavior getSpreadingBehavior(class_2248 block, BlockSpreadingType type) {
		return SPREADERS.get(block, type);
	}

	private static void setupVanillaBehavior() {
		addSpreaderBehavior(class_2246.field_10566, BlockSpreadingType.GRASS, ((state, level, pos) ->
				Blocks.GRASS_BLOCK.defaultBlockState()
						.setValue(BlockStateProperties.SNOWY, level.getBlockState(pos.above()).is(Blocks.SNOW))));

		addSpreaderBehavior(class_2246.field_10219, BlockSpreadingType.REVERT, ((state, level, pos) ->
				Blocks.DIRT.defaultBlockState()));

		addSpreaderBehavior(class_2246.field_10566, BlockSpreadingType.MYCELIUM, ((state, level, pos) ->
				Blocks.MYCELIUM.defaultBlockState()
						.setValue(BlockStateProperties.SNOWY, level.getBlockState(pos.above()).is(Blocks.SNOW))));

		addSpreaderBehavior(class_2246.field_10402, BlockSpreadingType.REVERT, ((state, level, pos) ->
				Blocks.DIRT.defaultBlockState()));

		addSpreaderBehavior(class_2246.field_10515, BlockSpreadingType.CRIMSON, ((state, level, pos) ->
				Blocks.CRIMSON_NYLIUM.defaultBlockState()));

		addSpreaderBehavior(class_2246.field_22120, BlockSpreadingType.REVERT, ((state, level, pos) ->
				Blocks.NETHERRACK.defaultBlockState()));

		addSpreaderBehavior(class_2246.field_10515, BlockSpreadingType.WARPED, ((state, level, pos) ->
				Blocks.WARPED_NYLIUM.defaultBlockState()));

		addSpreaderBehavior(class_2246.field_22113, BlockSpreadingType.REVERT, ((state, level, pos) ->
				Blocks.NETHERRACK.defaultBlockState()));
	}

	public static class SimpleSpreaderBehavior implements SpreadingBehavior {

		private final class_2680 state;

		public SimpleSpreaderBehavior(class_2680 state) {
			this.state = state;
		}

		@Override
		public class_2680 getStateForSpread(class_2680 state, class_1937 level, class_2338 pos) {
			return this.state;
		}
	}

}