package net.devtech.arrp.json.worldgen.biome;

import com.mojang.datafixers.util.Pair;
import com.mojang.serialization.Codec;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.DynamicOps;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import net.devtech.arrp.json.worldgen.JAttributeValue;

import java.util.*;

/**
 * Biome definition that matches modern JSON:
 * - Classic fields: has_precipitation, temperature, downfall, etc.
 * - "effects" for the remaining legacy visuals (water/grass/foliage colors).
 * - "attributes" for Environment Attributes (visual/*, gameplay/*, audio/*, etc.).
 *
 * No Gson anywhere; everything is driven by Mojang Codecs.
 */
public class JBiome {

	// =====================================================================
	// Root biome codec
	// =====================================================================

	public static final Codec<JBiome> CODEC = RecordCodecBuilder.create(instance ->
			instance.group(
					Codec.BOOL.optionalFieldOf("has_precipitation")
							.forGetter(JBiome::hasPrecipitationOptional),
					Codec.FLOAT.optionalFieldOf("temperature")
							.forGetter(JBiome::temperatureOptional),
					Codec.STRING.optionalFieldOf("temperature_modifier")
							.forGetter(JBiome::temperatureModifierOptional),
					Codec.FLOAT.optionalFieldOf("downfall")
							.forGetter(JBiome::downfallOptional),
					Effects.CODEC.optionalFieldOf("effects")
							.forGetter(JBiome::effectsOptional),
					Codec.unboundedMap(Codec.STRING, JAttributeValue.CODEC)
							.optionalFieldOf("attributes", Collections.emptyMap())
							.forGetter(b -> b.attributes == null ? Collections.emptyMap() : b.attributes),
					SpawnSettings.CODEC.optionalFieldOf("spawn_settings")
							.forGetter(JBiome::spawnSettingsOptional),
					Generation.CODEC.optionalFieldOf("generation")
							.forGetter(JBiome::generationOptional)
			).apply(instance, (precip, temp, tempMod, downfall, effects, attrs, spawn, gen) -> {
				JBiome biome = new JBiome();
				precip.ifPresent(biome::hasPrecipitation);
				temp.ifPresent(biome::temperature);
				tempMod.ifPresent(biome::temperatureModifier);
				downfall.ifPresent(biome::downfall);
				effects.ifPresent(biome::effects);
				biome.attributes(attrs);
				spawn.ifPresent(biome::spawnSettings);
				gen.ifPresent(biome::generation);
				return biome;
			})
	);

	// =====================================================================
	// Fields
	// =====================================================================

	private Boolean hasPrecipitation;
	private Float temperature;
	private String temperatureModifier;
	private Float downfall;

	private Effects effects;           // remaining legacy effect fields
	private Map<String, JAttributeValue> attributes;     // Environment Attributes
	private SpawnSettings spawnSettings;
	private Generation generation;

	// =====================================================================
	// Constructors / factories
	// =====================================================================

	public JBiome() {}

	public static JBiome biome() {
		return new JBiome();
	}

	/**
	 * Basic empty-biome template with everything null.
	 */
	public static JBiome empty() {
		return new JBiome();
	}

	// =====================================================================
	// Optional accessors for Codec
	// =====================================================================

	private Optional<Boolean> hasPrecipitationOptional() {
		return Optional.ofNullable(this.hasPrecipitation);
	}

	private Optional<Float> temperatureOptional() {
		return Optional.ofNullable(this.temperature);
	}

	private Optional<String> temperatureModifierOptional() {
		return Optional.ofNullable(this.temperatureModifier);
	}

	private Optional<Float> downfallOptional() {
		return Optional.ofNullable(this.downfall);
	}

	private Optional<Effects> effectsOptional() {
		return Optional.ofNullable(this.effects);
	}

	private Optional<Map<String, JAttributeValue>> attributesOptional() {
		return Optional.ofNullable(this.attributes);
	}

	private Optional<SpawnSettings> spawnSettingsOptional() {
		return Optional.ofNullable(this.spawnSettings);
	}

	private Optional<Generation> generationOptional() {
		return Optional.ofNullable(this.generation);
	}

	// =====================================================================
	// Builder-style setters
	// =====================================================================

	public JBiome hasPrecipitation(boolean has) {
		this.hasPrecipitation = has;
		return this;
	}

	public JBiome temperature(float temperature) {
		this.temperature = temperature;
		return this;
	}

	public JBiome temperatureModifier(String modifier) {
		this.temperatureModifier = modifier;
		return this;
	}

	public JBiome downfall(float downfall) {
		this.downfall = downfall;
		return this;
	}

	public JBiome effects(Effects effects) {
		this.effects = effects;
		return this;
	}

	public JBiome attributes(Map<String, JAttributeValue> attributes) {
		this.attributes = attributes;
		return this;
	}

	public JBiome spawnSettings(SpawnSettings spawnSettings) {
		this.spawnSettings = spawnSettings;
		return this;
	}

	public JBiome generation(Generation generation) {
		this.generation = generation;
		return this;
	}

	// =====================================================================
	// Convenience helpers
	// =====================================================================

	// ---- Effects helpers (remaining classic visual fields) ----

	private Effects ensureEffects() {
		if (this.effects == null) {
			this.effects = new Effects();
		}
		return this.effects;
	}

	// ---- Attributes helpers: raw map access ----

	private Map<String, JAttributeValue> ensureAttributes() {
		if (this.attributes == null) {
			this.attributes = new HashMap<>();
		}
		return this.attributes;
	}

	/** Generic attribute setter for string values. */
	public JBiome attribute(String id, String value) {
		ensureAttributes().put(id, JAttributeValue.ofString(value));
		return this;
	}

	/** Generic attribute setter for numeric values. */
	public JBiome attribute(String id, Number value) {
		ensureAttributes().put(id, JAttributeValue.ofFloat(value.floatValue()));
		return this;
	}

	/** Generic attribute setter for boolean values. */
	public JBiome attribute(String id, boolean value) {
		ensureAttributes().put(id, JAttributeValue.ofBoolean(value));
		return this;
	}

	// ---- Spawn helpers ----

	private SpawnSettings ensureSpawnSettings() {
		if (this.spawnSettings == null) {
			this.spawnSettings = new SpawnSettings();
		}
		return this.spawnSettings;
	}

	public JBiome spawnProbability(float probability) {
		ensureSpawnSettings().setCreatureSpawnProbability(probability);
		return this;
	}

	// =====================================================================
	// Nested types
	// =====================================================================

	/**
	 * Remaining biome "effects" fields after the Environment Attribute migration:
	 * - water_color
	 * - grass_color
	 * - foliage_color
	 * - grass_color_modifier
	 */
	public static class Effects implements Cloneable {
		private String waterColor;
		private String grassColor;
		private String foliageColor;
		private String grassColorModifier;

		public static final Codec<Effects> CODEC = RecordCodecBuilder.create(instance ->
				instance.group(
						Codec.STRING.optionalFieldOf("water_color").forGetter(e -> Optional.ofNullable(e.waterColor)),
						Codec.STRING.optionalFieldOf("grass_color").forGetter(e -> Optional.ofNullable(e.grassColor)),
						Codec.STRING.optionalFieldOf("foliage_color").forGetter(e -> Optional.ofNullable(e.foliageColor)),
						Codec.STRING.optionalFieldOf("grass_color_modifier").forGetter(e -> Optional.ofNullable(e.grassColorModifier))
				).apply(instance, (water, grass, foliage, modifier) -> {
					Effects e = new Effects();
					water.ifPresent(e::waterColor);
					grass.ifPresent(e::grassColor);
					foliage.ifPresent(e::foliageColor);
					modifier.ifPresent(e::grassColorModifier);
					return e;
				})
		);

		public Effects waterColor(String color) {
			this.waterColor = color;
			return this;
		}

		public Effects grassColor(String color) {
			this.grassColor = color;
			return this;
		}

		public Effects foliageColor(String color) {
			this.foliageColor = color;
			return this;
		}

		public Effects grassColorModifier(String modifier) {
			this.grassColorModifier = modifier;
			return this;
		}

		@Override
		public Effects clone() {
			try {
				return (Effects) super.clone();
			} catch (CloneNotSupportedException e) {
				throw new InternalError(e);
			}
		}
	}

	/**
	 * Environment Attributes map on the biome:
	 *
	 * "attributes": {
	 *   "minecraft:visual/fog_color": "#ffaa00",
	 *   "minecraft:gameplay/water_evaporates": true,
	 *   ...
	 * }
	 */
	public static class Attributes implements Cloneable {
		private final Map<String, AttributeValue> values = new LinkedHashMap<>();

		public static final Codec<Attributes> CODEC =
				Codec.unboundedMap(Codec.STRING, AttributeValue.CODEC)
						.xmap(Attributes::fromMap, Attributes::toMap);

		private static Attributes fromMap(Map<String, AttributeValue> map) {
			Attributes attrs = new Attributes();
			attrs.values.putAll(map);
			return attrs;
		}

		private Map<String, AttributeValue> toMap() {
			return new LinkedHashMap<>(this.values);
		}

		public Attributes putString(String key, String value) {
			if (key != null && value != null) {
				this.values.put(key, AttributeValue.ofString(value));
			}
			return this;
		}

		public Attributes putNumber(String key, double value) {
			if (key != null) {
				this.values.put(key, AttributeValue.ofNumber(value));
			}
			return this;
		}

		public Attributes putBoolean(String key, boolean value) {
			if (key != null) {
				this.values.put(key, AttributeValue.ofBoolean(value));
			}
			return this;
		}

		public Map<String, AttributeValue> getValues() {
			return Collections.unmodifiableMap(this.values);
		}

		public Attributes skyColorAttr(String hexRgb) {
			return putString("minecraft:visual/sky_color", hexRgb);
		}

		public Attributes fogColorAttr(String hexRgb) {
			return putString("minecraft:visual/fog_color", hexRgb);
		}

		public Attributes waterFogColorAttr(String hexRgb) {
			return putString("minecraft:visual/water_fog_color", hexRgb);
		}

		public Attributes waterFogEndDistance(float radius) {
			return putNumber("minecraft:visual/water_fog_start_distance", radius);
		}

		public Attributes waterFogStartDistance(float radius) {
			return putNumber("minecraft:visual/water_fog_end_distance", radius);
		}

		/** visual/cloud_opacity (0–1). */
		public Attributes cloudOpacityAttr(float opacity) {
			return putNumber("minecraft:visual/cloud_opacity", opacity);
		}

		/** visual/extra_fog (Nether-style dense fog toggle). */
		public Attributes extraFogAttr(boolean enabled) {
			return putBoolean("minecraft:visual/extra_fog", enabled);
		}

		/** gameplay/water_evaporates, etc., you can add more helpers like this. */
		public Attributes waterEvaporatesAttr(boolean evaporates) {
			return putBoolean("minecraft:gameplay/water_evaporates", evaporates);
		}


		@Override
		public Attributes clone() {
			try {
				Attributes clone = (Attributes) super.clone();
				clone.values.clear();
				clone.values.putAll(this.values);
				return clone;
			} catch (CloneNotSupportedException e) {
				throw new InternalError(e);
			}
		}
	}

	/**
	 * Value stored in the attributes map. We only support:
	 * - boolean
	 * - number
	 * - string (used e.g. for "#rrggbb" colors or IDs)
	 */
	public static abstract class AttributeValue {

		public static final Codec<AttributeValue> CODEC = new Codec<>() {
			@Override
			public <T> DataResult<T> encode(AttributeValue value, DynamicOps<T> ops, T prefix) {
				if (value instanceof BoolValue b) {
					return DataResult.success(ops.createBoolean(b.value));
				} else if (value instanceof NumberValue n) {
					return DataResult.success(ops.createDouble(n.value));
				} else if (value instanceof StringValue s) {
					return DataResult.success(ops.createString(s.value));
				}
				return DataResult.error(() -> "Unknown AttributeValue subclass: " + value.getClass());
			}

			@Override
			public <T> DataResult<Pair<AttributeValue, T>> decode(DynamicOps<T> ops, T input) {
				// try boolean
				Optional<Boolean> boolOpt = ops.getBooleanValue(input).result();
				if (boolOpt.isPresent()) {
					return DataResult.success(Pair.of(ofBoolean(boolOpt.get()), input));
				}

				// then number
				Optional<Number> numOpt = ops.getNumberValue(input).result();
				if (numOpt.isPresent()) {
					return DataResult.success(Pair.of(ofNumber(numOpt.get().doubleValue()), input));
				}

				// fallback to string
				Optional<String> strOpt = ops.getStringValue(input).result();
				if (strOpt.isPresent()) {
					return DataResult.success(Pair.of(ofString(strOpt.get()), input));
				}

				return DataResult.error(() -> "Unsupported attribute value (not bool/number/string)");
			}
		};

		public static AttributeValue ofBoolean(boolean value) {
			return new BoolValue(value);
		}

		public static AttributeValue ofNumber(double value) {
			return new NumberValue(value);
		}

		public static AttributeValue ofString(String value) {
			return new StringValue(value);
		}

		public static final class BoolValue extends AttributeValue {
			public final boolean value;
			public BoolValue(boolean value) { this.value = value; }
		}

		public static final class NumberValue extends AttributeValue {
			public final double value;
			public NumberValue(double value) { this.value = value; }
		}

		public static final class StringValue extends AttributeValue {
			public final String value;
			public StringValue(String value) { this.value = value; }
		}
	}

	public static class SpawnSettings implements Cloneable {
		private Float creatureSpawnProbability;

		public static final Codec<SpawnSettings> CODEC = RecordCodecBuilder.create(instance ->
				instance.group(
						Codec.FLOAT.optionalFieldOf("creature_spawn_probability")
								.forGetter(s -> Optional.ofNullable(s.creatureSpawnProbability))
				).apply(instance, probOpt -> {
					SpawnSettings s = new SpawnSettings();
					probOpt.ifPresent(s::setCreatureSpawnProbability);
					return s;
				})
		);

		public SpawnSettings setCreatureSpawnProbability(float probability) {
			this.creatureSpawnProbability = probability;
			return this;
		}

		public Float getCreatureSpawnProbability() {
			return creatureSpawnProbability;
		}

		@Override
		public SpawnSettings clone() {
			try {
				return (SpawnSettings) super.clone();
			} catch (CloneNotSupportedException e) {
				throw new InternalError(e);
			}
		}
	}

	public static class Generation implements Cloneable {
		private List<String> carvers = new ArrayList<>();
		private List<String> features = new ArrayList<>();

		public static final Codec<Generation> CODEC = RecordCodecBuilder.create(instance ->
				instance.group(
						Codec.STRING.listOf().optionalFieldOf("carvers", Collections.emptyList())
								.forGetter(g -> g.carvers == null ? Collections.emptyList() : g.carvers),
						Codec.STRING.listOf().optionalFieldOf("features", Collections.emptyList())
								.forGetter(g -> g.features == null ? Collections.emptyList() : g.features)
				).apply(instance, (carvers, features) -> {
					Generation g = new Generation();
					g.carvers = new ArrayList<>(carvers);
					g.features = new ArrayList<>(features);
					return g;
				})
		);

		public Generation addCarver(String id) {
			if (id != null) this.carvers.add(id);
			return this;
		}

		public Generation addFeature(String id) {
			if (id != null) this.features.add(id);
			return this;
		}

		public List<String> getCarvers() {
			return Collections.unmodifiableList(carvers);
		}

		public List<String> getFeatures() {
			return Collections.unmodifiableList(features);
		}

		@Override
		public Generation clone() {
			try {
				Generation clone = (Generation) super.clone();
				clone.carvers = new ArrayList<>(this.carvers);
				clone.features = new ArrayList<>(this.features);
				return clone;
			} catch (CloneNotSupportedException e) {
				throw new InternalError(e);
			}
		}
	}
}
