package net.devtech.arrp.json.iteminfo.model;

import com.google.gson.JsonObject;
import com.mojang.datafixers.util.Either;
import com.mojang.serialization.*;

import java.util.List;
import java.util.Optional;

/**
 * Represents a case in a select model.
 */
public class JSelectCase implements Cloneable {
    private Object when; // Can be a single value or a list of values
    private JItemModel model;

    public JSelectCase() {}

    public JSelectCase(Object when, JItemModel model) {
        this.when = when;
        this.model = model;
    }

    public static JSelectCase of(Object when, JItemModel model) {
        return new JSelectCase(when, model);
    }

    // Getters and Setters
    public Object getWhen() {
        return when;
    }

    public JItemModel getModel() {
        return model;
    }

    public JsonObject toJson() {
        JsonObject json = new JsonObject();
        if (when != null) {
			switch (when) {
				case String s -> json.addProperty("when", s);
				case Number i -> json.addProperty("when", i);
				case Boolean b -> json.addProperty("when", b);
				default -> throw new IllegalArgumentException("Unsupported type for 'when'");
			}
        }
        if (model != null) {
            json.add("model", model.toJson());
        }
        return json;
    }

    @Override
    public JSelectCase clone() {
        JSelectCase cloned = new JSelectCase();
        cloned.when = this.when;
        cloned.model = (this.model != null) ? this.model.clone() : null;
        return cloned;
    }

    // ---- Codecs ----

    /** Single primitive (string/number/boolean). */
    private static final Codec<Object> PRIMITIVE = Codec.of(
            // Encoder
            new Encoder<>() {
                @Override public <T> DataResult<T> encode(Object v, DynamicOps<T> ops, T prefix) {
                    if (v instanceof String s)   return DataResult.success(ops.createString(s));
                    if (v instanceof Boolean b)  return DataResult.success(ops.createBoolean(b));
                    if (v instanceof Number n) {
                        if (n instanceof Integer || n instanceof Short || n instanceof Byte)
                            return DataResult.success(ops.createInt(n.intValue()));
                        if (n instanceof Long)
                            return DataResult.success(ops.createLong(n.longValue()));
                        if (n instanceof Float)
                            return DataResult.success(ops.createFloat(n.floatValue()));
                        return DataResult.success(ops.createDouble(n.doubleValue()));
                    }
                    return DataResult.error(() -> "JSelectCase.when: unsupported value type: " + v.getClass());
                }
            },
            // Decoder
            new Decoder<>() {
                @Override public <T> DataResult<com.mojang.datafixers.util.Pair<Object,T>> decode(DynamicOps<T> ops, T input) {
                    // try boolean → number → string
                    Optional<Boolean> b = ops.getBooleanValue(input).result();
                    if (b.isPresent()) return DataResult.success(com.mojang.datafixers.util.Pair.of(b.get(), input));

                    Optional<Number> num = ops.getNumberValue(input).result();
                    if (num.isPresent()) {
                        Number n = num.get();
                        double d = n.doubleValue();
                        if (Math.rint(d) == d) {
                            long L = n.longValue();
                            Object boxed = (L >= Integer.MIN_VALUE && L <= Integer.MAX_VALUE) ? (int)L : L;
                            return DataResult.success(com.mojang.datafixers.util.Pair.of(boxed, input));
                        }
                        return DataResult.success(com.mojang.datafixers.util.Pair.of(d, input));
                    }

                    Optional<String> s = ops.getStringValue(input).result();
					return s.<DataResult<com.mojang.datafixers.util.Pair<Object, T>>>map(string ->
                            DataResult.success(com.mojang.datafixers.util.Pair.of(string, input))
                    ).orElseGet(() -> DataResult.error(() -> "JSelectCase.when: expected string/number/boolean"));
				}
            }
    );

    /** Accept single primitive or array of primitives; keep `Object` as value-or-list. */
    private static final Codec<Object> WHEN_CODEC =
            Codec.either(PRIMITIVE, PRIMITIVE.listOf())
                    .xmap(
                            e -> e.map(v -> v, l -> l), // Either -> Object
                            o -> (o instanceof List<?> l) ? Either.right((List<Object>) l) : Either.left(o)
                    );

    public static final Codec<JSelectCase> CODEC =
            com.mojang.serialization.codecs.RecordCodecBuilder.create(i -> i.group(
                    WHEN_CODEC.fieldOf("when").forGetter(JSelectCase::getWhen),
                    JItemModel.CODEC.fieldOf("model").forGetter(JSelectCase::getModel)
            ).apply(i, JSelectCase::new));
}
